import type { Request } from 'express';
import { HTTPStatusCode } from 'tachyon-type-library';
import type {
  RedirectCommand,
  RedirectManager,
} from '../redirectManagerExpressAdapter';

export interface PageAlias {
  /**
   * A list of regexes that match the route patterns you wish to redirect. These
   * regexes must use named capture groups to identify relevant parts of the
   * path-pattern for re-use in building the new redirect path; this is both
   * serve as better documentation of intent and also protect against
   * index-based misses.
   */
  aliases: RegExp[];
  /**
   * A function that accepts the named capture groups from an alias regex match
   * and uses them to build the new redirect path.
   */
  pathBuilder: (
    groups: Required<RegExpMatchArray>['groups'],
  ) => string | undefined;
}

export interface PageAliasMiddlewareOpts {
  pageAliases: PageAlias[];
}

export function pageAliasManager({
  pageAliases,
}: PageAliasMiddlewareOpts): RedirectManager {
  return ({ path }: Request): RedirectCommand => {
    let location: string | undefined;
    loop: for (const { aliases, pathBuilder } of pageAliases) {
      for (const alias of aliases) {
        const match = path.match(alias);
        if (match) {
          location = pathBuilder(match.groups || {});
          break loop;
        }
      }
    }

    if (location) {
      return {
        location,
        statusCode: HTTPStatusCode.Found,
      };
    }
  };
}
