import type { Request } from 'express';
import { HTTPStatusCode } from 'tachyon-type-library';
import type { PageAlias } from '.';
import { pageAliasManager } from '.';

describe(pageAliasManager, () => {
  // tests modelled on tmw's needs for now to aid pattern development
  const pageAliases: PageAlias[] = [
    // channel page
    {
      aliases: [
        /^\/(?<login>\w+)\/mobile\/?$/i,
        /^\/(?<login>\w+)\/rooms\/\w+\/?$/i,
        /^\/(?<login>\w+)\/squad\/?$/i,
      ],
      pathBuilder: ({ login }) => `/${login}`,
    },
    // channel profile page
    {
      aliases: [
        /^\/(?<login>\w+)\/about\/?$/i,
        /^\/(?<login>\w+)\/clips\/?$/i,
        /^\/(?<login>\w+)\/collections?(?:\/\w+)?\/?$/i,
        /^\/(?<login>\w+)\/events\/?$/i,
        /^\/(?<login>\w+)\/followers\/?$/i,
        /^\/(?<login>\w+)\/following\/?$/i,
        /^\/(?<login>\w+)\/profile\/\w+\/?$/i,
        /^\/(?<login>\w+)\/schedule\/?$/i,
        /^\/(?<login>\w+)\/videos\/?$/i,
        /^\/(?<login>\w+)\/videos\/all\/?$/i,
        // old video format urls
        /^\/(?<login>\w+)\/(?:a|b|c|streams)\/\w+\/?$/i,
      ],
      pathBuilder: ({ login }) => `/${login}/profile`,
    },
    // vod page
    {
      aliases: [
        /^\/\w+\/video\/(?<videoId>\w+)\/?/i,
        /^\/videos\/(?<videoId>\w+)\/\w+\/?/i,
        /^\/video\/(?<videoId>\w+)\/?/i,
      ],
      pathBuilder: ({ videoId }) => `/videos/${videoId}`,
    },
    // not found page
    {
      aliases: [/^\/events\/\w+\/?/i],
      pathBuilder: () => `/not-found`,
    },
  ];

  const configuredManager = pageAliasManager({ pageAliases });

  interface RedirectTestCase {
    location: string;
    path: string;
  }

  it.each`
    path                                        | location
    ${'/monstercat/mobile'}                     | ${'/monstercat'}
    ${'/monstercat/Mobile'}                     | ${'/monstercat'}
    ${'/monstercat/rooms/someRoom'}             | ${'/monstercat'}
    ${'/monstercat/squad'}                      | ${'/monstercat'}
    ${'/monstercat/about'}                      | ${'/monstercat/profile'}
    ${'/monstercat/ABOUT'}                      | ${'/monstercat/profile'}
    ${'/monstercat/clips'}                      | ${'/monstercat/profile'}
    ${'/monstercat/collection'}                 | ${'/monstercat/profile'}
    ${'/monstercat/collection/YMu4W3SklxXlfQ'}  | ${'/monstercat/profile'}
    ${'/monstercat/collections'}                | ${'/monstercat/profile'}
    ${'/monstercat/collections/YMu4W3SklxXlfQ'} | ${'/monstercat/profile'}
    ${'/monstercat/events'}                     | ${'/monstercat/profile'}
    ${'/monstercat/followers'}                  | ${'/monstercat/profile'}
    ${'/monstercat/following'}                  | ${'/monstercat/profile'}
    ${'/monstercat/profile/random'}             | ${'/monstercat/profile'}
    ${'/monstercat/schedule'}                   | ${'/monstercat/profile'}
    ${'/monstercat/videos'}                     | ${'/monstercat/profile'}
    ${'/monstercat/videos/all'}                 | ${'/monstercat/profile'}
    ${'/monstercat/a/12345'}                    | ${'/monstercat/profile'}
    ${'/monstercat/b/12345'}                    | ${'/monstercat/profile'}
    ${'/monstercat/c/12345'}                    | ${'/monstercat/profile'}
    ${'/monstercat/streams/12345'}              | ${'/monstercat/profile'}
    ${'/monstercat/video/12345'}                | ${'/videos/12345'}
    ${'/videos/12345/chat'}                     | ${'/videos/12345'}
    ${'/video/12345'}                           | ${'/videos/12345'}
    ${'/events/12345'}                          | ${'/not-found'}
  `('$path redirects to $location', ({ location, path }: RedirectTestCase) => {
    expect(
      configuredManager({
        path,
      } as Request),
    ).toEqual({
      location,
      statusCode: HTTPStatusCode.Found,
    });
    expect(
      configuredManager({
        path: `${path}/`,
      } as Request),
    ).toEqual({
      location,
      statusCode: HTTPStatusCode.Found,
    });
  });

  interface NonredirectTestCase {
    path: string;
  }

  // paths handled in-app
  it.each`
    path
    ${'/monstercat'}
    ${'/monstercat/profile'}
    ${'/monstercat/clip/Clip'}
    ${'/clip/Clip'}
    ${'/videos/12345'}
    ${'/directory'}
    ${'/directory/all'}
    ${'/directory/game/Fortnite'}
  `('$path does not redirect', ({ path }: NonredirectTestCase) => {
    expect(
      configuredManager({
        path,
      } as Request),
    ).toBeUndefined();
  });
});
