import type { Request, Response } from 'express';
import { RedirectionCode } from 'tachyon-type-library';
import { redirectManagerExpressAdapter } from '.';

describe('redirectManagerExpressAdapter', () => {
  let callOrder: string[] = [];
  const ALIAS_PATH = '/foo/bar';
  const TARGET = '/monstercat';
  const MOCK_REDIRECT = jest.fn(() => callOrder.push('redirect'));
  const MOCK_END = jest.fn(() => callOrder.push('end'));
  const NEXT = jest.fn(() => callOrder.push('next'));
  const MOCK_RESPONSE = jest.fn(() => ({
    end: MOCK_END,
    redirect: MOCK_REDIRECT,
  }));
  const STATUS_CODE = RedirectionCode.SeeOther;

  afterEach(() => {
    callOrder = [];
  });

  describe('when commanded to redirect', () => {
    beforeEach(() => {
      redirectManagerExpressAdapter(() => ({
        location: TARGET,
        statusCode: STATUS_CODE,
      }))(
        { path: ALIAS_PATH } as Request,
        new MOCK_RESPONSE() as any as Response,
        NEXT,
      );
    });

    it('redirects an alias to the original', () => {
      expect(MOCK_REDIRECT).toHaveBeenCalledWith(STATUS_CODE, `${TARGET}`);
    });

    it('calls end on the response', () => {
      expect(MOCK_END).toHaveBeenCalled();
    });

    it('does not call next', () => {
      expect(NEXT).not.toHaveBeenCalled();
    });

    it('calls redirect, end, & next in the right order', () => {
      expect(callOrder).toEqual(['redirect', 'end']);
    });
  });

  describe('not commanded to redirect', () => {
    beforeEach(() => {
      redirectManagerExpressAdapter(() => undefined)(
        { path: '/monstercat' } as Request,
        new MOCK_RESPONSE() as any as Response,
        NEXT,
      );
    });

    it('does not call redirect', () => {
      expect(MOCK_REDIRECT).not.toHaveBeenCalled();
    });

    it('does not call end', () => {
      expect(MOCK_END).not.toHaveBeenCalled();
    });

    it('does still calls next', () => {
      expect(NEXT).toHaveBeenCalled();
    });
  });
});
