import type { Request } from 'express';
import type { Query } from 'express-serve-static-core';
import minimatch from 'minimatch';
import type { StringifiableRecord } from 'query-string';
import { HTTPStatusCode } from 'tachyon-type-library';
import type {
  RedirectCommand,
  RedirectManager,
} from '../redirectManagerExpressAdapter';
import { prepareDesktopUrl } from './prepareDesktopUrl';

export interface RedirectToDesktopWebMiddlewareOpts {
  baseRedirectQueryParameters?: Record<string, string>;
  pathDepthCutoff?: number;
  pathsToForceRedirect: string[];
  redirectPreservedQueryParameters?: string[];
  wasRedirectedFromDesktopQueryParam?: string;
}

/**
 * Checks for cases in which we should redirect to desktop web. This can be for
 * redundancy as the CDN can be set to handle some of these cases as well.
 *
 * Handled redirect cases:
 * - send users to Desktop for specific routes (CDN redundancy)
 * - send users to Desktop when a path with many too parts is requested (and they were redirected by the CDN)
 */
export function redirectToDesktopWebManager({
  baseRedirectQueryParameters = {},
  pathDepthCutoff,
  pathsToForceRedirect,
  redirectPreservedQueryParameters = [],
  wasRedirectedFromDesktopQueryParam,
}: RedirectToDesktopWebMiddlewareOpts): RedirectManager {
  function isForcedRedirectPath(path: string): boolean {
    return pathsToForceRedirect.some((forcedPath) => {
      return minimatch(path, forcedPath, { nocase: true });
    });
  }

  function isUncertainDestination(request: Request): boolean {
    if (!pathDepthCutoff) {
      return false;
    }

    const pathIsTooDeep =
      (request.path.match(/\//g) || []).length >= pathDepthCutoff;
    // This query param will be present when the CDN redirects a user who
    // was attempting to visit desktop from a mobile device.
    const wasRedirectedFromDesktopWeb =
      !wasRedirectedFromDesktopQueryParam ||
      request.query[wasRedirectedFromDesktopQueryParam] === 'true';

    return pathIsTooDeep && wasRedirectedFromDesktopWeb;
  }

  return (request: Request): RedirectCommand => {
    if (
      !isForcedRedirectPath(request.path.slice(1)) &&
      !isUncertainDestination(request)
    ) {
      return;
    }

    const params: Query = {
      ...baseRedirectQueryParameters,
    };

    redirectPreservedQueryParameters.forEach((param) => {
      if (request.query[param]) {
        params[param] = request.query[param];
      }
    });

    const redirect = {
      location: prepareDesktopUrl(request.path, params as StringifiableRecord),
      statusCode: HTTPStatusCode.Found,
    };

    return redirect;
  };
}
