import type { Request } from 'express';
import { random } from 'faker';
import { HTTPStatusCode } from 'tachyon-type-library';
import type { RedirectToDesktopWebMiddlewareOpts } from '.';
import { redirectToDesktopWebManager } from '.';

describe(redirectToDesktopWebManager, () => {
  const desktopRedirectParamKey = 'redirect-to-desktop';
  const baseRedirectQueryParameters = {
    [desktopRedirectParamKey]: 'true',
  };
  const desktopRedirectParamString = `${desktopRedirectParamKey}=true`;

  const pathDepthCutoff = 4;

  const pathsToForceRedirect = [
    'exact',
    'startOne/*',
    'startMany/**',
    '*/endOne',
    '**/endMany',
  ];

  const redirectPreservedQueryParameters = ['tt_yolo'];

  const wasRedirectedFromDesktopQueryParam = 'im-from-desktop';

  const fullConfig: Required<RedirectToDesktopWebMiddlewareOpts> = {
    baseRedirectQueryParameters,
    pathDepthCutoff,
    pathsToForceRedirect,
    redirectPreservedQueryParameters,
    wasRedirectedFromDesktopQueryParam,
  };

  const configuredMiddleware = redirectToDesktopWebManager(fullConfig);

  describe('requests for forced redirect paths', () => {
    it('redirects to desktop for exact match', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/exact',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/exact?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('redirects to desktop for exact match with case difference', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/EXACT',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/EXACT?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('passes along preserved params when redirecting to desktop', () => {
      const tt_yolo = random.alphaNumeric();
      const tt_foreveralone = random.alphaNumeric();
      const middlewareResponse = configuredMiddleware({
        path: '/exact',
        query: { tt_foreveralone, tt_yolo },
      } as any as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/exact?${desktopRedirectParamString}&tt_yolo=${tt_yolo}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('redirects to desktop even when coming from desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/exact',
        query: { [wasRedirectedFromDesktopQueryParam]: 'true' },
      } as any as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/exact?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('redirects a forced path with exact trailing depth to desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/startOne/foo',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/startOne/foo?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('does not redirect a depthless path matching the start of a forced path with exact trailing depth', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/startOne',
        query: {},
      } as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('does not redirect a deeper trailing path that starts with a forced path with exact trailing depth', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/startOne/foo/bar',
        query: {},
      } as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('redirects a forced path with unlimited trailing depth to desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/startMany/foo/bar',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/startMany/foo/bar?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('does not redirect a depthless trailing path matching the start of a forced path with unlimited trailing depth', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/startMany',
        query: {},
      } as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('redirects a forced path with exact leading depth to desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/foo/endOne',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/foo/endOne?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('does not redirect a depthless path matching the end of a forced path with exact leading depth', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/endOne',
        query: {},
      } as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('does not redirect a deeper leading path that ends with a forced path with exact leading depth', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/foo/bar/endOne',
        query: {},
      } as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('redirects a forced path with unlimited leading depth to desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/foo/bar/endMany',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/foo/bar/endMany?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('redirects a depthless leading path matching the end of a forced path with unlimited leading depth', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/endMany',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/endMany?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });
  });

  describe('when the paths are too deep', () => {
    it('redirects if the request came from desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/1/2/3/4',
        query: { [wasRedirectedFromDesktopQueryParam]: 'true' },
      } as any as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/1/2/3/4?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('does not redirect if the request did not come from desktop', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/1/2/3/4',
        query: {},
      } as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('does not redirect if path is not too deep', () => {
      const middlewareResponse = configuredMiddleware({
        path: '/1/2/3',
        query: { [wasRedirectedFromDesktopQueryParam]: 'true' },
      } as any as Request);

      expect(middlewareResponse).toBeUndefined();
    });
  });

  describe('partial configurations', () => {
    it('missing pathDepthCutoff means no length-based redirects', () => {
      const partialConfig: RedirectToDesktopWebMiddlewareOpts = {
        ...fullConfig,
      };
      delete partialConfig.pathDepthCutoff;

      const middlewareResponse = redirectToDesktopWebManager(partialConfig)({
        path: '/1/2/3/4/5/6/7/8',
        query: { [wasRedirectedFromDesktopQueryParam]: 'true' },
      } as any as Request);

      expect(middlewareResponse).toBeUndefined();
    });

    it('missing wasRedirectedFromDesktopQueryParam means all length-based redirect without query param', () => {
      const partialConfig: RedirectToDesktopWebMiddlewareOpts = {
        ...fullConfig,
      };
      delete partialConfig.wasRedirectedFromDesktopQueryParam;

      const middlewareResponse = redirectToDesktopWebManager(partialConfig)({
        path: '/1/2/3/4',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/1/2/3/4?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('missing baseRedirectQueryParameters means no extra query params added', () => {
      const partialConfig: RedirectToDesktopWebMiddlewareOpts = {
        ...fullConfig,
      };
      delete partialConfig.baseRedirectQueryParameters;

      const middlewareResponse = redirectToDesktopWebManager(partialConfig)({
        path: '/exact',
        query: {},
      } as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/exact`,
        statusCode: HTTPStatusCode.Found,
      });
    });

    it('missing redirectPreservedQueryParameters means no params preserved', () => {
      const partialConfig: RedirectToDesktopWebMiddlewareOpts = {
        ...fullConfig,
      };
      delete partialConfig.redirectPreservedQueryParameters;

      const tt_yolo = random.alphaNumeric();
      const tt_foreveralone = random.alphaNumeric();
      const middlewareResponse = redirectToDesktopWebManager(partialConfig)({
        path: '/exact',
        query: { tt_foreveralone, tt_yolo },
      } as any as Request);

      expect(middlewareResponse).toEqual({
        location: `https://www.twitch.tv/exact?${desktopRedirectParamString}`,
        statusCode: HTTPStatusCode.Found,
      });
    });
  });
});
