import type {
  AnimationEvent,
  FormEvent,
  KeyboardEvent,
  MouseEvent,
} from 'react';
import type { PartialDeep } from 'type-fest';

/**
 * Generates mock change events for use in testing, designed for maximum
 * flexibility. The first arg is mirrored to both `checked` and `value` for both
 * `target` and `currentTarget` (and blur() is mocked for both objects since
 * that is a common need). The optional second arg allows you pass in other
 * attributes that you might need to merge into the event payload as well. This
 * also supports generics to enable generating the correct type of FormEvents,
 * though it defaults to `HTMLInputElement`. For instance, if you need to make
 * `FormEvent<HTMLSelectElement>`, then you would do
 * `mockChangeEvent<HTMLSelectElement>(...)`.
 */
export function mockChangeEvent<T extends Element = HTMLInputElement>(
  value: boolean | number | string,
  overrides: PartialDeep<FormEvent<T>> = {},
): FormEvent<T> {
  return {
    ...overrides,
    currentTarget: {
      blur: jest.fn(),
      checked: value,
      value,
      ...(overrides.currentTarget ?? {}),
    },
    target: {
      blur: jest.fn(),
      checked: value,
      value,
      ...(overrides.target ?? {}),
    },
  } as unknown as FormEvent<T>;
}

export function mockKeyDown<T extends Element = HTMLInputElement>(
  opts: PartialDeep<KeyboardEvent<T>>,
): KeyboardEvent<T> {
  return opts as unknown as KeyboardEvent<T>;
}

export function mockInputBlurEvent<
  T extends Element = HTMLInputElement,
>(): FormEvent<T> {
  return {} as FormEvent<T>;
}

export function mockInputFocusEvent<
  T extends Element = HTMLInputElement,
>(): FormEvent<T> {
  return {} as FormEvent<T>;
}

export function mockClickEvent<
  T extends Element = HTMLButtonElement,
>(): MouseEvent<T> {
  return {} as MouseEvent<T>;
}

export function mockAnimationEvent<
  T extends Element = HTMLDivElement,
>(): AnimationEvent<T> {
  return {} as AnimationEvent<T>;
}
