import type {
  MountRendererProps,
  ReactWrapper,
  ShallowRendererProps,
  ShallowWrapper,
} from 'enzyme';
import { mount, shallow } from 'enzyme';
import type { ComponentClass, Context, FC, PropsWithChildren } from 'react';
import { defaultsDeep } from 'tachyon-utils-stdlib';
import type { PartialDeep } from 'type-fest';

// context doesn't work in enzyme shallow yet, so this is only in mount tests for now
// https://github.com/enzymejs/enzyme/issues/2189
export type ContextWithValue = [context: Context<any>, value: any];
type WithWrappingContexts<FactoryOpts> = FactoryOpts & {
  wrappingContexts?: () => ContextWithValue[];
};
type ContextValueMap = Map<ContextWithValue[0], ContextWithValue[1]>;

/**
 * Wrapper factory args for components with no props
 */
type WrapperFactoryArgsForNoProps<FactoryOpts> = [] | [undefined, FactoryOpts];
/**
 * Wrapper factory args for components with at least 1 required prop
 */
type WrapperFactoryArgsForSomeRequiredProps<P, FactoryOpts> =
  | [() => P, FactoryOpts]
  | [() => P];
/**
 * Wrapper factory args for components with only optional props
 */
type WrapperFactoryArgsForOnlyOptionalProps<P, FactoryOpts> =
  | WrapperFactoryArgsForNoProps<FactoryOpts>
  | WrapperFactoryArgsForSomeRequiredProps<P, FactoryOpts>;
/**
 * Wrapper factory args varying based on shape of a component's props
 */
type WrapperFactoryArgs<P, FactoryOpts> = {} extends P
  ? {} extends Required<P>
    ? WrapperFactoryArgsForNoProps<FactoryOpts>
    : WrapperFactoryArgsForOnlyOptionalProps<P, FactoryOpts>
  : WrapperFactoryArgsForSomeRequiredProps<P, FactoryOpts>;

/**
 * A shallow wrapper bundled with the props used to construct the component
 */
type ShallowWrapperBundle<P, S> = {
  props: PropsWithChildren<P>;
  wrapper: ShallowWrapper<P, S>;
};
/**
 * The factory function for creating shallow test setups
 */
type ShallowWrapperFactory<P, S> = (
  customProps?: PropsWithChildren<PartialDeep<P>>,
) => ShallowWrapperBundle<P, S>;

type UpdateWrappingContext = (...update: ContextWithValue) => void;
/**
 * The factory function for creating shallow test setups
 */
type MountWrapperBundle<P, S> = {
  contexts: ContextValueMap;
  props: PropsWithChildren<P>;
  updateWrappingContext: UpdateWrappingContext;
  wrapper: ReactWrapper<P, S>;
};
type MountWrapperFactoryOps = {
  wrappingContexts?: ContextWithValue[];
};
/**
 * A mount wrapper bundled with the props and context used to construct the
 * component
 */
type MountWrapperFactory<P, S> = (
  customProps?: PropsWithChildren<PartialDeep<P>>,
  opts?: MountWrapperFactoryOps,
) => MountWrapperBundle<P, S>;

/**
 * Creates an enzyme shallow wrapper factory for a component. It takes
 * a default prop factory function for the component, and returns a function that
 * will generate the shallow wrapper combined with any props you pass in.
 * Props will be deeply merged with the default props to allow the bare minimum
 * of custom props for a given test. The combined set of props will also be
 * returned.
 *
 * To send default or custom children to your component, just include them in
 * the relevant props object. If your component requires children, you should
 * declare their shape (even if the shape is just ReactNode) in the component's
 * props.
 *
 * It also takes an optional opts object for passing in shallow renderer options
 * such as disableLifecycleMethods.
 *
 * Example Use:
 *
 * const setup = createShallowWrapperFactory(Foo, () => ({ bar: 'someValue', baz: { boink: true } }));
 * const { props, wrapper } = setup({ baz: { boink: false} });
 * props === { bar: 'someValue', baz: { boink: false } }
 *
 * @param Component the component to be wrapped
 * @param propGenerator a function that generates a default prop set for Component
 * @param opts takes shallow renderer options
 * @return shallow wrapper generator function that takes optional customProps
 * and customChildren
 */
export function createShallowWrapperFactory<P, S = {}>(
  Component: ComponentClass<P, S> | FC<P>,
  ...[propGenerator, opts = {}]: WrapperFactoryArgs<
    PropsWithChildren<P>,
    ShallowRendererProps
  >
): ShallowWrapperFactory<P, S> {
  return (
    customProps = {} as PropsWithChildren<PartialDeep<P>>,
  ): ShallowWrapperBundle<P, S> => {
    // if there's no propGenerator there are no required props, equivalent to {}
    const defaultProps = propGenerator
      ? propGenerator()
      : ({} as PropsWithChildren<P>);
    // ensure only a single set of children to avoid merges
    if (customProps.children) {
      delete defaultProps.children;
    }
    const combinedProps = defaultsDeep(
      defaultProps,
      customProps,
    ) as PropsWithChildren<P>;

    return {
      props: combinedProps,
      wrapper: shallow<P, S>(<Component {...combinedProps} />, opts),
    };
  };
}

/**
 * Creates an enzyme mount wrapper factory for a component. It takes
 * a default prop factory function for the component, and returns a function that
 * will generate the mount wrapper combined with any props you pass in.
 * Props will be deeply merged with the default props to allow the bare minimum
 * of custom props for a given test. The combined set of props will also be returned.
 *
 * To send default or custom children to your component, just include them in
 * the relevant props object. If your component requires children, you should
 * declare their shape (even if the shape is just ReactNode) in the component's
 * props.
 *
 * It also takes an optional opts object for passing in mount renderer options
 * such as context.
 *
 * Example Use:
 *
 * const setup = createMountWrapperFactory(Foo, () => ({ bar: 'someValue', baz: { boink: true } }));
 * const { props, wrapper } = setup({ baz: { boink: false} });
 * // props === { bar: 'someValue', baz: { boink: true } }
 *
 * @param Component the component to be wrapped
 * @param propGenerator a function that generates a default prop set for Component
 * @param opts takes mount renderer options enhanced with `wrappingContexts` for providing context values
 * @return shallow wrapper generator function that takes optional customProps, customMocks,
 * and customChildren
 */
export function createMountWrapperFactory<P, S = {}>(
  Component: ComponentClass<P, S> | FC<P>,
  ...[propGenerator, opts = {}]: WrapperFactoryArgs<
    PropsWithChildren<P>,
    WithWrappingContexts<MountRendererProps>
  >
): MountWrapperFactory<P, S> {
  const defaultWrappingContexts = opts.wrappingContexts;

  return (
    customProps = {} as PropsWithChildren<PartialDeep<P>>,
    runOpts = {},
  ): MountWrapperBundle<P, S> => {
    // dupe opts to prevent side-effects across tests
    const mountOpts = { ...opts };
    // if there's no propGenerator there are no required props, equivalent to {}
    const defaultProps = propGenerator
      ? propGenerator()
      : ({} as PropsWithChildren<P>);
    // ensure only a single set of children to avoid merges
    if (customProps.children) {
      delete defaultProps.children;
    }
    const combinedProps = defaultsDeep(
      defaultProps,
      customProps,
    ) as PropsWithChildren<P>;

    const customWrappingContexts = runOpts.wrappingContexts ?? [];
    // we use a Map here to perform a dedupe of any contexts that are set in
    // both default and custom args, with the custom override winning
    const combinedWrappingContexts: ContextValueMap = new Map([
      ...(defaultWrappingContexts?.() ?? []),
      ...customWrappingContexts,
    ]);

    mountOpts.wrappingComponent = ({ children, ...props }) =>
      Array.from(combinedWrappingContexts.entries()).reduce(
        (Acc, [context, value]) => (
          <context.Provider children={Acc} value={value} />
        ),
        opts.wrappingComponent ? (
          <opts.wrappingComponent {...props} children={children} />
        ) : (
          <>{children}</>
        ),
      );

    const wrapper = mount<P, S>(<Component {...combinedProps} />, mountOpts);

    const updateWrappingContext: UpdateWrappingContext = (
      ...update: ContextWithValue
    ) => {
      combinedWrappingContexts.set(...update);
      // update() is not available here, so force a re-render without changing props
      wrapper.getWrappingComponent().setProps({});
    };

    return {
      contexts: combinedWrappingContexts,
      props: combinedProps,
      updateWrappingContext,
      wrapper,
    };
  };
}
