import type { FC } from 'react';
import { Component, createContext, useContext } from 'react';
import { createMountWrapperFactory, createShallowWrapperFactory } from '.';

const TEST_SELECTOR = { 'data-test-selector': 'provided-component' };

interface TestProps {
  bar: {
    baz: boolean;
    boink: number;
  };
  foo: string;
}

const TestFCWithRequiredProps: FC<TestProps> = (props) => {
  return <div {...TEST_SELECTOR} {...props} />;
};

class TestCCWithRequiredProps extends Component<TestProps> {
  public override render(): JSX.Element {
    return <div {...TEST_SELECTOR} {...this.props} />;
  }
}

interface OptionalProps {
  foo?: {
    bar: string;
  };
}

const TestFCWithoutRequiredProps: FC<OptionalProps> = (props) => {
  return <div {...TEST_SELECTOR} {...props} />;
};

class TestCCWithoutRequiredProps extends Component<OptionalProps> {
  public override render(): JSX.Element {
    return <div {...TEST_SELECTOR} {...this.props} />;
  }
}

describe('Test Wrapper Factories', () => {
  // The output of this is stable across runs so that we can ensure that prop
  // merging works as intended. Normally, you'd want to use Faker to generate
  // random values for fuzzing.
  const propGenerator = (): TestProps => ({
    bar: {
      baz: false,
      boink: 42,
    },
    foo: 'Rock and Roll',
  });

  describe(createShallowWrapperFactory, () => {
    const setup = createShallowWrapperFactory(
      TestCCWithRequiredProps,
      propGenerator,
    );

    it('returns a wrapper with the props applied from the factory function', () => {
      const { props, wrapper } = setup();
      expect(props).toEqual(propGenerator());

      const divWrapper = wrapper.find(TEST_SELECTOR);
      expect(divWrapper).toHaveLength(1);
      expect(divWrapper.props()).toEqual({
        ...props,
        ...TEST_SELECTOR,
        children: undefined,
      });
    });

    it('works with FunctionComponents', () => {
      const setupWithFC = createShallowWrapperFactory(
        TestFCWithRequiredProps,
        propGenerator,
      );

      const { props, wrapper } = setupWithFC();
      expect(props).toEqual(propGenerator());

      const divWrapper = wrapper.find(TEST_SELECTOR);
      expect(divWrapper).toHaveLength(1);
      expect(divWrapper.props()).toEqual({
        ...propGenerator(),
        ...TEST_SELECTOR,
        children: undefined,
      });
    });

    describe('when a component needs to render children', () => {
      const propsWithChildrenGenerator = () => ({
        ...propGenerator(),
        children: <TestFCWithRequiredProps {...propGenerator()} />,
      });

      const setupWithChildren = createShallowWrapperFactory(
        TestCCWithRequiredProps,
        propsWithChildrenGenerator,
      );

      it('renders children from default props', () => {
        const { wrapper } = setupWithChildren();
        expect(wrapper.find(TestFCWithRequiredProps)).toHaveLength(1);
      });

      it('renders children from custom props', () => {
        const { wrapper } = setup({
          children: <TestFCWithRequiredProps {...propGenerator()} />,
        });
        expect(wrapper.find(TestFCWithRequiredProps)).toHaveLength(1);
      });

      it('renders children from custom props (overriding default props children without merging)', () => {
        const childProps = { className: 'override' };
        const { wrapper } = setupWithChildren({
          children: <div {...childProps} />,
        });

        expect(wrapper.find(TestFCWithRequiredProps)).toHaveLength(0);
        const overridingChild = wrapper.find('.override');
        expect(overridingChild).toHaveLength(1);
        expect(overridingChild.props()).toEqual(childProps);
      });
    });

    describe('when custom props are provided to the wrapper factory', () => {
      it('deep merges the props and applies them to the Component', () => {
        const { props, wrapper } = setup({
          bar: { baz: true },
          foo: 'R&B',
        });

        expect(props).toEqual({
          bar: {
            baz: true,
            boink: 42,
          },
          foo: 'R&B',
        });

        const divWrapper = wrapper.find(TEST_SELECTOR);
        expect(divWrapper).toHaveLength(1);
        expect(divWrapper.props()).toEqual({
          ...props,
          ...TEST_SELECTOR,
          children: undefined,
        });
      });
    });

    describe('when the setup has no default required props', () => {
      it('works with Function Components', () => {
        const noPropSetup = createShallowWrapperFactory(
          TestFCWithoutRequiredProps,
        );
        const { wrapper } = noPropSetup({ foo: { bar: '42' } });
        expect(wrapper.find(TEST_SELECTOR).props()).toEqual({
          foo: { bar: '42' },
          ...TEST_SELECTOR,
          children: undefined,
        });
      });

      it('works with Class Components', () => {
        const noPropSetup = createShallowWrapperFactory(
          TestCCWithoutRequiredProps,
        );
        const { wrapper } = noPropSetup({ foo: { bar: '42' } });
        expect(wrapper.find(TEST_SELECTOR).props()).toEqual({
          foo: { bar: '42' },
          ...TEST_SELECTOR,
          children: undefined,
        });
      });

      it('works with a prop generator', () => {
        const propSetup = createShallowWrapperFactory(
          TestFCWithoutRequiredProps,
          () => ({ foo: { bar: '42' } }),
        );
        const { wrapper } = propSetup();
        expect(wrapper.find(TEST_SELECTOR).props()).toEqual({
          foo: { bar: '42' },
          ...TEST_SELECTOR,
          children: undefined,
        });
      });
    });
  });

  describe(createMountWrapperFactory, () => {
    const setup = createMountWrapperFactory(
      TestCCWithRequiredProps,
      propGenerator,
    );

    it('returns a wrapper with the props applied from the factory function', () => {
      const { props, wrapper } = setup();
      expect(props).toEqual(propGenerator());

      const divWrapper = wrapper.find(TEST_SELECTOR);
      expect(divWrapper).toHaveLength(1);
      expect(divWrapper.props()).toEqual({
        ...props,
        ...TEST_SELECTOR,
        children: undefined,
      });
    });

    it('works with FunctionComponents', () => {
      const setupWithFC = createMountWrapperFactory(
        TestFCWithRequiredProps,
        propGenerator,
      );

      const { props, wrapper } = setupWithFC();
      expect(props).toEqual(propGenerator());

      const divWrapper = wrapper.find(TEST_SELECTOR);
      expect(divWrapper).toHaveLength(1);
      expect(divWrapper.props()).toEqual({
        ...propGenerator(),
        ...TEST_SELECTOR,
        children: undefined,
      });
    });

    describe('when a component needs to render children', () => {
      const propsWithChildrenGenerator = () => ({
        ...propGenerator(),
        children: <TestFCWithRequiredProps {...propGenerator()} />,
      });

      const setupWithChildren = createMountWrapperFactory(
        TestCCWithRequiredProps,
        propsWithChildrenGenerator,
      );

      it('renders children from default props', () => {
        const { wrapper } = setupWithChildren();
        expect(wrapper.find(TestFCWithRequiredProps)).toHaveLength(1);
      });

      it('renders children from custom props', () => {
        const { wrapper } = setup({
          children: <TestFCWithRequiredProps {...propGenerator()} />,
        });
        expect(wrapper.find(TestFCWithRequiredProps)).toHaveLength(1);
      });

      it('renders children from custom props (overriding default props children without merging)', () => {
        const childProps = { className: 'override' };
        const { wrapper } = setupWithChildren({
          children: <div {...childProps} />,
        });

        expect(wrapper.find(TestFCWithRequiredProps)).toHaveLength(0);
        const overridingChild = wrapper.find('.override');
        expect(overridingChild).toHaveLength(1);
        expect(overridingChild.props()).toEqual(childProps);
      });
    });

    describe('when custom props are provided to the wrapper factory', () => {
      it('deep merges the props and applies them to the Component', () => {
        const { props, wrapper } = setup({ bar: { baz: true }, foo: 'R&B' });

        expect(props).toEqual({
          bar: {
            baz: true,
            boink: 42,
          },
          foo: 'R&B',
        });

        const divWrapper = wrapper.find(TEST_SELECTOR);
        expect(divWrapper).toHaveLength(1);
        expect(divWrapper.props()).toEqual({
          ...props,
          ...TEST_SELECTOR,
          children: undefined,
        });
      });
    });

    describe('when the setup has no default required props', () => {
      it('works with Function Components', () => {
        const noPropSetup = createMountWrapperFactory(
          TestFCWithoutRequiredProps,
        );
        const { wrapper } = noPropSetup({ foo: { bar: '42' } });
        expect(wrapper.find(TEST_SELECTOR).props()).toEqual({
          foo: { bar: '42' },
          ...TEST_SELECTOR,
          children: undefined,
        });
      });

      it('works with Class Components', () => {
        const noPropSetup = createMountWrapperFactory(
          TestCCWithoutRequiredProps,
        );
        const { wrapper } = noPropSetup({ foo: { bar: '42' } });
        expect(wrapper.find(TEST_SELECTOR).props()).toEqual({
          foo: { bar: '42' },
          ...TEST_SELECTOR,
          children: undefined,
        });
      });

      it('works with a prop generator', () => {
        const propSetup = createMountWrapperFactory(
          TestFCWithoutRequiredProps,
          () => ({ foo: { bar: '42' } }),
        );
        const { wrapper } = propSetup();
        expect(wrapper.find(TEST_SELECTOR).props()).toEqual({
          foo: { bar: '42' },
          ...TEST_SELECTOR,
          children: undefined,
        });
      });
    });

    describe('with mocked contexts', () => {
      const foo1Context = createContext(1);
      const useFoo1Context = () => useContext(foo1Context);
      const foo2Context = createContext(1);
      const useFoo2Context = () => useContext(foo2Context);

      describe('function components', () => {
        const TestFC = () => {
          const foo1 = useFoo1Context();
          const foo2 = useFoo2Context();
          return <div>{foo1 + foo2}</div>;
        };
        const setupWithoutDefaultContextMock =
          createMountWrapperFactory(TestFC);

        it('baselines with no context mocks', () => {
          const { wrapper } = setupWithoutDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('2');
        });

        it('handles a single custom context mock', () => {
          const { wrapper } = setupWithoutDefaultContextMock(undefined, {
            wrappingContexts: [[foo1Context, 5]],
          });
          expect(wrapper.find('div').text()).toEqual('6');
        });

        it('handles multiple custom context mocks', () => {
          const { wrapper } = setupWithoutDefaultContextMock(undefined, {
            wrappingContexts: [
              [foo1Context, 10],
              [foo2Context, 10],
            ],
          });
          expect(wrapper.find('div').text()).toEqual('20');
        });

        it('handles single default context mock', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestFC,
            undefined,
            {
              wrappingContexts: () => [[foo2Context, 15]],
            },
          );

          const { wrapper } = setupWithDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('16');
        });

        it('handles multiple default context mocks', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestFC,
            undefined,
            {
              wrappingContexts: () => [
                [foo1Context, 20],
                [foo2Context, 20],
              ],
            },
          );

          const { wrapper } = setupWithDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('40');
        });

        it('handles custom context mocks overriding default context mocks and returns utilized contexts', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestFC,
            undefined,
            {
              wrappingContexts: () => [
                [foo1Context, 25],
                [foo2Context, 25],
              ],
            },
          );

          const { contexts, wrapper } = setupWithDefaultContextMock(undefined, {
            wrappingContexts: [[foo1Context, 100]],
          });
          const output = wrapper.find('div').text();
          const expectedOutput = Array.from(contexts.values()).reduce(
            (acc, val) => acc + val,
          );
          // matches human expectations
          expect(output).toEqual('125');
          // matches exhaustive computed expections
          expect(output).toEqual(expectedOutput.toString());
          expect(contexts.get(foo1Context)).toEqual(100);
          expect(contexts.get(foo2Context)).toEqual(25);
        });

        it('does not clobber a wrapping component and properly forwards props', () => {
          const wrappingComponent: FC<{ value: number }> = ({
            children,
            value,
          }) => <foo1Context.Provider children={children} value={value} />;
          const setupWithWrappingComponent = createMountWrapperFactory(
            TestFC,
            undefined,
            {
              wrappingComponent,
              wrappingComponentProps: { value: 33 },
            },
          );

          const { contexts, wrapper } = setupWithWrappingComponent(undefined, {
            wrappingContexts: [[foo2Context, 33]],
          });
          expect(wrapper.find('div').text()).toEqual('66');
          expect(contexts.has(foo1Context)).toBeFalsy();
          expect(contexts.get(foo2Context)).toEqual(33);
        });

        it('responds to updates to a custom context', () => {
          const { updateWrappingContext, wrapper } =
            setupWithoutDefaultContextMock(undefined, {
              wrappingContexts: [[foo1Context, 44]],
            });
          expect(wrapper.find('div').text()).toEqual('45');

          updateWrappingContext(foo1Context, 55);

          expect(wrapper.find('div').text()).toEqual('56');
        });

        it('responds to updates to a default context', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestFC,
            undefined,
            {
              wrappingContexts: () => [[foo1Context, 66]],
            },
          );

          const { updateWrappingContext, wrapper } =
            setupWithDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('67');

          updateWrappingContext(foo1Context, 77);

          expect(wrapper.find('div').text()).toEqual('78');
        });

        it('responds to updates to an unmocked context when previously using context fallback value', () => {
          const { updateWrappingContext, wrapper } =
            setupWithoutDefaultContextMock();

          expect(wrapper.find('div').text()).toEqual('2');

          updateWrappingContext(foo1Context, 88);

          expect(wrapper.find('div').text()).toEqual('89');
        });
      });

      describe('class components', () => {
        class TestCC extends Component {
          public override render(): JSX.Element {
            return (
              <foo1Context.Consumer>
                {(foo1) => (
                  <foo2Context.Consumer>
                    {(foo2) => <div>{foo1 + foo2}</div>}
                  </foo2Context.Consumer>
                )}
              </foo1Context.Consumer>
            );
          }
        }

        const setupWithoutDefaultContextMock =
          createMountWrapperFactory(TestCC);

        it('baselines with no context mocks', () => {
          const { wrapper } = setupWithoutDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('2');
        });

        it('handles a single custom context mock', () => {
          const { wrapper } = setupWithoutDefaultContextMock(undefined, {
            wrappingContexts: [[foo1Context, 5]],
          });
          expect(wrapper.find('div').text()).toEqual('6');
        });

        it('handles multiple custom context mocks', () => {
          const { wrapper } = setupWithoutDefaultContextMock(undefined, {
            wrappingContexts: [
              [foo1Context, 10],
              [foo2Context, 10],
            ],
          });
          expect(wrapper.find('div').text()).toEqual('20');
        });

        it('handles single default context mock', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestCC,
            undefined,
            {
              wrappingContexts: () => [[foo2Context, 15]],
            },
          );

          const { wrapper } = setupWithDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('16');
        });

        it('handles multiple default context mocks', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestCC,
            undefined,
            {
              wrappingContexts: () => [
                [foo1Context, 20],
                [foo2Context, 20],
              ],
            },
          );

          const { wrapper } = setupWithDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('40');
        });

        it('handles custom context mocks overriding default context mocks', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestCC,
            undefined,
            {
              wrappingContexts: () => [
                [foo1Context, 25],
                [foo2Context, 25],
              ],
            },
          );

          const { contexts, wrapper } = setupWithDefaultContextMock(undefined, {
            wrappingContexts: [[foo1Context, 100]],
          });
          const output = wrapper.find('div').text();
          const expectedOutput = Array.from(contexts.values()).reduce(
            (acc, val) => acc + val,
          );
          // matches human expectations
          expect(output).toEqual('125');
          // matches exhaustive computed expections
          expect(output).toEqual(expectedOutput.toString());
          expect(contexts.get(foo1Context)).toEqual(100);
          expect(contexts.get(foo2Context)).toEqual(25);
        });

        it('does not clobber a wrapping component and properly forwards props', () => {
          const wrappingComponent: FC<{ value: number }> = ({
            children,
            value,
          }) => <foo1Context.Provider children={children} value={value} />;
          const setupWithWrappingComponent = createMountWrapperFactory(
            TestCC,
            undefined,
            {
              wrappingComponent,
              wrappingComponentProps: { value: 33 },
            },
          );

          const { contexts, wrapper } = setupWithWrappingComponent(undefined, {
            wrappingContexts: [[foo2Context, 33]],
          });
          expect(wrapper.find('div').text()).toEqual('66');
          expect(contexts.has(foo1Context)).toBeFalsy();
          expect(contexts.get(foo2Context)).toEqual(33);
        });

        it('responds to updates to a custom context', () => {
          const { updateWrappingContext, wrapper } =
            setupWithoutDefaultContextMock(undefined, {
              wrappingContexts: [[foo1Context, 44]],
            });
          expect(wrapper.find('div').text()).toEqual('45');

          updateWrappingContext(foo1Context, 55);

          expect(wrapper.find('div').text()).toEqual('56');
        });

        it('responds to updates to a default context', () => {
          const setupWithDefaultContextMock = createMountWrapperFactory(
            TestCC,
            undefined,
            {
              wrappingContexts: () => [[foo1Context, 66]],
            },
          );

          const { updateWrappingContext, wrapper } =
            setupWithDefaultContextMock();
          expect(wrapper.find('div').text()).toEqual('67');

          updateWrappingContext(foo1Context, 77);

          expect(wrapper.find('div').text()).toEqual('78');
        });

        it('responds to updates to an unmocked context when previously using context fallback value', () => {
          const { updateWrappingContext, wrapper } =
            setupWithoutDefaultContextMock();

          expect(wrapper.find('div').text()).toEqual('2');

          updateWrappingContext(foo1Context, 88);

          expect(wrapper.find('div').text()).toEqual('89');
        });
      });
    });
  });
});
