/**
 * Sets all top-level keys within an object to be NonNullable
 */
export type NonNullableObj<Obj extends {}> = {
  [key in keyof Obj]: NonNullable<Obj[key]>;
};

/**
 * Deeply sets all keys within an object to be NonNullable
 */
export type DeepNonNullableObj<Obj extends {}> = {
  [key in keyof Obj]: NonNullable<Obj[key]> extends {}
    ? DeepNonNullableObj<NonNullable<Obj[key]>>
    : NonNullable<Obj[key]>;
};

/**
 * Removes all keys from the FullObj that are present in the FilteringObj
 */
export type Diff<FullObj, FilteringObj> = {
  [RetainedKey in Exclude<
    keyof FullObj,
    keyof FilteringObj
  >]: FullObj[RetainedKey];
};

/**
 * Declares an Array with atleast one element
 */
export type NonEmptyArray<T> = [T, ...T[]];

/**
 * Extracts a literal union of the keys whose values are of the specified type
 */
export type KeysForValuesOfType<Obj extends {}, ValueType> = {
  [Key in keyof Obj]-?: Obj[Key] extends ValueType ? Key : never;
}[keyof Obj];

/**
 * Distributes omit over a union
 * 'BetterOmit' from https://github.com/microsoft/TypeScript/issues/31501
 */
export type DistributedOmit<Union, Keys extends keyof any> = Union extends any
  ? Omit<Union, Keys>
  : never;

/**
 * Any function. Useful for extending from with additional properties or methods.
 */
export interface AnyFunction extends Function {
  (...args: any[]): any;
}

/**
 * A cancelable function.
 */
export interface CancelableFunction extends AnyFunction {
  cancel(): void;
}

/**
 * Recursive version of "Mutable" from type-fest.
 */
export type MutableDeep<T> = { -readonly [P in keyof T]: MutableDeep<T[P]> };

/**
 * Type for forcing consumers to provide a generics value for types that can
 * otherwise default to `{}` (which would have no type constraints).
 *
 * For example: `foo<CriticalType extends {} = ForceExplicitGenericsObject>() {}`
 * would force all consumers of `foo()` to provide an explicit generics argument.
 */
export type ForceExplicitGenericsObject = {
  requiresExplicitGenericsForSafety: never;
};
