import type { Options as MemOpts } from 'mem';
import mem from 'mem';
import QuickLRU from 'quick-lru';

// forces cacheKey to be mandatory
interface MemoizeCustomOpts<Args extends unknown[], Ret>
  extends Omit<MemOpts<Args, unknown, Ret>, 'cacheKey'>,
    Required<Pick<MemOpts<Args, unknown, Ret>, 'cacheKey'>> {}

export type MemoizeCustomAllOpts<Args extends unknown[], Ret> = Omit<
  MemoizeCustomOpts<Args, Ret>,
  'cache'
>;
/**
 * A memoization function that uses the supplied cacheKey generation function to
 * build a cache key and memoize all calls to the wrapped function. The cache
 * key generator should be as efficient as possible based on the specifics of
 * the wrapped function's arguments.
 *
 * Ideally one of the more specific memoizer functions in this package which
 * supply their own cacheKey functions.
 */
export function memoizeCustomAll<Args extends unknown[], Ret>(
  fn: (...args: Args) => Ret,
  opts: MemoizeCustomAllOpts<Args, Ret>,
): (...args: Args) => Ret {
  return mem(fn, { ...opts });
}

export interface MemoizeCustomSomeOpts<Args extends unknown[], Ret>
  extends Omit<MemoizeCustomOpts<Args, Ret>, 'cache'> {
  maxSize: number;
}
/**
 * A memoization function that uses the supplied cacheKey generation function to
 * build a cache key and memoize the most recent N unique (according to the
 * cache key generation function) calls to the wrapped function, with N governed
 * my the maxSize opt value. The cache key generator should be as efficient as
 * possible based on the specifics of the wrapped function's arguments.
 *
 * Ideally one of the more specific memoizer functions in this package which
 * supply their own cacheKey functions.
 */
export function memoizeCustomSome<Args extends unknown[], Ret>(
  fn: (...args: Args) => Ret,
  { maxSize, ...opts }: MemoizeCustomSomeOpts<Args, Ret>,
): (...args: Args) => Ret {
  return mem(fn, { ...opts, cache: new QuickLRU({ maxSize }) });
}
