import type { KeysForValuesOfType, NonEmptyArray } from 'tachyon-type-library';
import type { Primitive } from 'type-fest';
import type {
  MemoizeCustomAllOpts,
  MemoizeCustomSomeOpts,
} from '../memoizeCustom';
import { memoizeCustomAll, memoizeCustomSome } from '../memoizeCustom';

type KeysForPrimitiveValues<Obj extends {}> = KeysForValuesOfType<
  Obj,
  Primitive
>;

type ArgsWithLeadObj<T extends {} = {}> = [T, ...unknown[]];

function leadObjCacheKey<T extends {}>(
  targetKeys: NonEmptyArray<KeysForPrimitiveValues<T>>,
): (args: ArgsWithLeadObj<T>) => string {
  return (args: ArgsWithLeadObj<T>) =>
    targetKeys.reduce<string>((acc, key) => acc + args[0][key], '');
}

export interface MemoizeLeadObjKeysAllOpts<Args extends ArgsWithLeadObj, Ret>
  extends Omit<MemoizeCustomAllOpts<Args, Ret>, 'cacheKey'> {
  targetKeys: NonEmptyArray<KeysForPrimitiveValues<Args[0]>>;
}
/**
 * A memoization function that uses the value of specified keys from the object
 * that is the first argument to a function to build a cache key and memoize all
 * calls to the wrapped function. The target keys must have primitive values,
 * which is enforced by the types. This can be very performant since cache key
 * calculation is tightly constrained, but care must be taken to identify the
 * relevant keys to avoid discarding important argument changes.
 */
export function memoizeLeadObjKeysAll<Args extends ArgsWithLeadObj, Ret>(
  fn: (...args: Args) => Ret,
  { targetKeys, ...opts }: MemoizeLeadObjKeysAllOpts<Args, Ret>,
): (...args: Args) => Ret {
  return memoizeCustomAll(fn, {
    ...opts,
    cacheKey: leadObjCacheKey(targetKeys),
  });
}

export interface MemoizeLeadObjKeysSomeOpts<Args extends ArgsWithLeadObj, Ret>
  extends Omit<MemoizeCustomSomeOpts<Args, Ret>, 'cacheKey'> {
  targetKeys: NonEmptyArray<KeysForPrimitiveValues<Args[0]>>;
}
/**
 * A memoization function that uses the value of specified keys from the object
 * that is the first argument to a function to build a cache key and memoize the
 * most recent N unique (according to the specified keys) calls to the wrapped
 * function, with N governed my the maxSize opt value. The target keys must have
 * primitive values, which is enforced by the types. This can be very performant
 * since cache key calculation is tightly constrained, but care must be taken to
 * identify the relevant keys to avoid discarding important argument changes.
 */
export function memoizeLeadObjKeysSome<Args extends ArgsWithLeadObj, Ret>(
  fn: (...args: Args) => Ret,
  { targetKeys, ...opts }: MemoizeLeadObjKeysSomeOpts<Args, Ret>,
): (...args: Args) => Ret {
  return memoizeCustomSome(fn, {
    ...opts,
    cacheKey: leadObjCacheKey(targetKeys),
  });
}
