import { memoizeLeadObjKeysAll, memoizeLeadObjKeysSome } from '.';

describe('memoize lead object keys', () => {
  interface ObjFuncOpts {
    a: number;
    b: number;
    c: number;
    d: number;
  }
  function objFunc({ a, b, c, d }: ObjFuncOpts): number {
    return a + b + c + d;
  }
  const mockObjFunc = jest.fn(objFunc);

  describe(memoizeLeadObjKeysAll, () => {
    it('caches the return value of multiple function calls based on target keys', () => {
      const memoObjFunc = memoizeLeadObjKeysAll(
        mockObjFunc as any as typeof objFunc,
        {
          targetKeys: ['a', 'b'],
        },
      );
      expect(memoObjFunc({ a: 1, b: 2, c: 3, d: 4 })).toEqual(10);
      expect(memoObjFunc({ a: 1, b: 2, c: 3, d: 4 })).toEqual(10);
      expect(mockObjFunc).toHaveBeenCalledTimes(1);
      // target keys unchanged
      expect(memoObjFunc({ a: 1, b: 2, c: 0, d: 0 })).toEqual(10);
      expect(mockObjFunc).toHaveBeenCalledTimes(1);
      // one target key changed
      expect(memoObjFunc({ a: 5, b: 2, c: 3, d: 4 })).toEqual(14);
      expect(memoObjFunc({ a: 5, b: 2, c: 3, d: 4 })).toEqual(14);
      expect(mockObjFunc).toHaveBeenCalledTimes(2);
      // both target keys changed
      expect(memoObjFunc({ a: 5, b: 6, c: 7, d: 8 })).toEqual(26);
      expect(memoObjFunc({ a: 5, b: 6, c: 7, d: 8 })).toEqual(26);
      expect(mockObjFunc).toHaveBeenCalledTimes(3);
    });
  });

  describe(memoizeLeadObjKeysSome, () => {
    it('caches the return value of last N unique function calls based on target keys', () => {
      const memoObjFunc = memoizeLeadObjKeysSome(
        mockObjFunc as any as typeof objFunc,
        {
          maxSize: 1,
          targetKeys: ['a', 'b'],
        },
      );

      expect(memoObjFunc({ a: 1, b: 2, c: 3, d: 4 })).toEqual(10);
      expect(memoObjFunc({ a: 1, b: 2, c: 3, d: 4 })).toEqual(10);
      expect(mockObjFunc).toHaveBeenCalledTimes(1);
      expect(memoObjFunc({ a: 5, b: 6, c: 7, d: 8 })).toEqual(26);
      expect(memoObjFunc({ a: 5, b: 6, c: 7, d: 8 })).toEqual(26);
      expect(mockObjFunc).toHaveBeenCalledTimes(2);
      expect(memoObjFunc({ a: 1, b: 2, c: 3, d: 4 })).toEqual(10);
      expect(memoObjFunc({ a: 1, b: 2, c: 3, d: 4 })).toEqual(10);
      expect(mockObjFunc).toHaveBeenCalledTimes(3);
    });
  });
});
