import type { NonEmptyArray } from 'tachyon-type-library';
import type {
  MemoizeCustomAllOpts,
  MemoizeCustomSomeOpts,
} from '../memoizeCustom';
import { memoizeCustomAll, memoizeCustomSome } from '../memoizeCustom';

export type MemoizeSimpleAllOpts<
  Args extends NonEmptyArray<unknown>,
  Ret,
> = Omit<MemoizeCustomAllOpts<Args, Ret>, 'cacheKey'>;
/**
 * A memoization function that uses the first argument to a function to build a
 * cache key (creating shallow comparison between the key) and memoize all calls
 * to the function. This is super fast but works best on primitives; care should
 * be taken to avoid attempting to use this on functions that take objects that
 * are freshly created with each invocation as these will not be shallowly
 * equal.
 */
export function memoizeSimpleAll<Args extends NonEmptyArray<unknown>, Ret>(
  fn: (...args: Args) => Ret,
  opts: MemoizeSimpleAllOpts<Args, Ret> = {},
): (...args: Args) => Ret {
  return memoizeCustomAll(fn, { ...opts, cacheKey: (args) => args[0] });
}

export type MemoizeSimpleSomeOpts<
  Args extends NonEmptyArray<unknown>,
  Ret,
> = Omit<MemoizeCustomSomeOpts<Args, Ret>, 'cacheKey'>;
/**
 * A memoization function that uses the first argument to a function to build a
 * cache key (creating shallow comparison between the key) and memoize the most
 * recent N unique (according to the first arg's shallow equality) calls to the
 * wrapped function, with N governed my the maxSize opt value. This is super
 * fast but works best on primitives; care should be taken to avoid attempting
 * to use this on functions that take objects that are freshly created with each
 * invocation as these will not be shallowly equal.
 */
export function memoizeSimpleSome<Args extends NonEmptyArray<unknown>, Ret>(
  fn: (...args: Args) => Ret,
  opts: MemoizeSimpleSomeOpts<Args, Ret>,
): (...args: Args) => Ret {
  return memoizeCustomSome(fn, { ...opts, cacheKey: (args) => args[0] });
}
