import { memoizeSimpleAll, memoizeSimpleSome } from '.';

describe('memoize simple', () => {
  const mockFn = jest.fn((...args: unknown[]) => JSON.stringify(args));

  describe(memoizeSimpleAll, () => {
    it('caches the return value of multiple function calls based on first arg shallow equality', () => {
      const testObj = { b: 2 };
      const memoFn = memoizeSimpleAll(mockFn);
      // primitive
      expect(memoFn(1)).toEqual('[1]');
      expect(memoFn(1)).toEqual('[1]');
      expect(mockFn).toHaveBeenCalledTimes(1);
      // object with persistent reference
      expect(memoFn(testObj)).toEqual('[{"b":2}]');
      expect(memoFn(testObj)).toEqual('[{"b":2}]');
      expect(mockFn).toHaveBeenCalledTimes(2);
      // equal-but-unique objects
      expect(memoFn({ b: 2 })).toEqual('[{"b":2}]');
      expect(memoFn({ b: 2 })).toEqual('[{"b":2}]');
      expect(mockFn).toHaveBeenCalledTimes(4);
      // untracked second arg
      expect(memoFn(3, 4)).toEqual('[3,4]');
      expect(memoFn(3, 4)).toEqual('[3,4]');
      expect(memoFn(3, 5)).toEqual('[3,4]');
      expect(mockFn).toHaveBeenCalledTimes(5);
    });
  });

  describe(memoizeSimpleSome, () => {
    it('caches the return value of last N unique function calls with complex arguments', () => {
      const memoFn = memoizeSimpleSome(mockFn, { maxSize: 1 });
      expect(memoFn(1)).toEqual('[1]');
      expect(memoFn(1)).toEqual('[1]');
      expect(mockFn).toHaveBeenCalledTimes(1);
      expect(memoFn(2)).toEqual('[2]');
      expect(memoFn(2)).toEqual('[2]');
      expect(mockFn).toHaveBeenCalledTimes(2);
      expect(memoFn(1)).toEqual('[1]');
      expect(memoFn(1)).toEqual('[1]');
      expect(mockFn).toHaveBeenCalledTimes(3);
    });
  });
});
