# tachyon-utils-react

Custom React functionality for SSR.

`yarn add tachyon-utils-react`

## Available Hooks

- useAsync
- useCallbackMemoizedByKey
- [useConst](#useconst)
- [useConstCallback](#useconstcallback)
- useDebounce
- useDebouncedState
- useDelayed
- useEffectOnce
- useEvent
- useForceUpdate
- useInterval
- useJSEnabled
- [useLatest](#uselatest)
- [useMethods](#usemethods)
- useMountedState
- usePoll
- usePrevious
- useStorage
- useThrottle
- useTimeout
- useUnmount

## Adding New react-use Hooks

If you'd like to add another `react-use` hook to this package, reach out in
#tachyon-lib-support so we can ensure it is safe to use and compatible with the
Tachyon platform.

## Usage Notes

### useConst

Returns a stable value on each render based on the value passed to the hook or
the value returned by calling the passed function on initial render. Any
argument changes beyond the first render are ignored.

### useConstCallback

Convenience helper for returning stable callbacks which is equivalent to calling
`useConst(() => callback)`. This is also similar to what the built-in
`useCallback` provides when the dependency list is empty, however, `useMemo` and
`useCallback` don't guarantee a stable value even with an empty dependency list
but this version does.

### useLatest

Version of
[useLatest](https://github.com/streamich/react-use/blob/master/docs/useLatest.md)
which is safe for future concurrent mode rendering. Instead of updating the
`ref`'s current value during render, it is updated in the commit phase via
`useLayoutEffect`.

This has the minor side effect of the `.current` property actually being the
previous value _during render_ except on the first render, but if the current
value is needed during render the value passed to this hook can simply be used
elsewhere directly.

### useMethods

Similar to
[useMethods](https://github.com/streamich/react-use/blob/master/docs/useMethods.md)
from `react-use` but the create factory function is called only once and is
passed a dispatch function which behaves like the setter returned from
`useState`. This allows the use of async methods which dispatch multiple times
without making the simple cases any more complex (just call dispatch in place of
a return).

Additionally, this version supports an optional 3rd argument whose value will be
saved to a `ref` and passed as a second parameter to the create function,
allowing the returned methods to access the current value without needing to
explicitly push this value into state (thus saving a re-render).

The `react-use` example would look like this:

```tsx
import type { MethodDispatch } from 'tachyon-utils-react';
import { useMethods } from 'tachyon-utils-react';

const initialState = {
  count: 0,
};

function createMethods(dispatch: MethodDispatch<typeof initialState>) {
  return {
    reset() {
      dispatch(initialState);
    },
    increment() {
      dispatch((state) => ({ ...state, count: state.count + 1 }));
    },
    decrement() {
      dispatch((state) => ({ ...state, count: state.count - 1 }));
    },
  };
}

const Demo = () => {
  const [state, methods] = useMethods(createMethods, initialState);

  return (
    <>
      <p>Count: {state.count}</p>
      <button onClick={methods.decrement}>-</button>
      <button onClick={methods.increment}>+</button>
    </>
  );
};
```
