import { useEffect, useState } from 'react';
import { CANCELLED_PROMISE_ERROR_MESSAGE } from 'tachyon-utils-stdlib';

interface PendingState {
  error: undefined;
  status: 'pending';
  value: undefined;
}

interface RejectedState {
  error: Error;
  status: 'rejected';
  value: undefined;
}

interface FulfilledState<T> {
  error: undefined;
  status: 'fulfilled';
  value: T;
}

export type AsyncState<T> = FulfilledState<T> | PendingState | RejectedState;

const INITIAL_STATE = {
  error: undefined,
  status: 'pending',
  value: undefined,
} as const;

/**
 * Calls an async function one time and manages the async life cycle. If the
 * function reference changes, the new function will be invoked.
 *
 * If the return Promise is rejected with new Error('cancelled'), setState will
 * not be called in the catch handler and the state will remain 'pending'. This
 * is useful for managing cleanup durring unmounting.
 *
 * @param {Function} fn The async function
 */
export function useAsync<Value>(fn: () => Promise<Value>): AsyncState<Value> {
  const [state, setState] = useState(INITIAL_STATE as AsyncState<Value>);

  // We only want to rerun if fn ref changes
  /* eslint-disable react-hooks/exhaustive-deps */
  useEffect(() => {
    // fn ref changed
    if (state !== INITIAL_STATE) {
      setState(INITIAL_STATE);
    }

    fn()
      .then((value: Value) => {
        setState({
          error: undefined,
          status: 'fulfilled',
          value,
        });
      })
      .catch((error: Error) => {
        if (error.message !== CANCELLED_PROMISE_ERROR_MESSAGE) {
          setState({ error, status: 'rejected', value: undefined });
        }
      });
  }, [fn]);

  return state;
}
