import { renderHook } from '@testing-library/react-hooks';
import { CANCELLED_PROMISE_ERROR_MESSAGE } from 'tachyon-utils-stdlib';
import { useAsync } from '.';

describe(useAsync, () => {
  it('initially returns pending state', async () => {
    const fn = () => new Promise(() => null);
    const { result } = renderHook(() => useAsync(fn));

    expect(result.current).toEqual({
      error: undefined,
      status: 'pending',
      value: undefined,
    });
  });

  it('returns success state', async () => {
    const fn = async () => 1;
    const { result, waitForNextUpdate } = renderHook(() => useAsync(fn));
    await waitForNextUpdate();

    expect(result.current).toEqual({
      error: undefined,
      status: 'fulfilled',
      value: 1,
    });
  });

  it('returns error state', async () => {
    const fn = async () => {
      throw new Error('uhoh');
    };
    const { result, waitForNextUpdate } = renderHook(() => useAsync(fn));
    await waitForNextUpdate();

    expect(result.current).toEqual({
      error: new Error('uhoh'),
      status: 'rejected',
      value: undefined,
    });
  });

  it('does not update state if Promise is "cancelled"', async () => {
    const error = {
      get message() {
        return CANCELLED_PROMISE_ERROR_MESSAGE;
      },
    };
    const spy = jest.spyOn(error, 'message', 'get');
    const fn = jest.fn(async () => {
      throw error;
    });
    const { result } = renderHook(() => useAsync(fn));
    // wait for cancelled status to settle
    await Promise.resolve().then(() => true);
    expect(spy).toHaveBeenCalled();

    expect(result.current).toEqual({
      error: undefined,
      status: 'pending',
      value: undefined,
    });
  });

  it('invokes new fn when fn ref changes', async () => {
    let fn = async () => 1;
    const { rerender, result, waitForNextUpdate } = renderHook(() =>
      useAsync(fn),
    );

    // initial
    expect(result.current).toEqual({
      error: undefined,
      status: 'pending',
      value: undefined,
    });

    // async fn resolves
    await waitForNextUpdate();
    expect(result.current).toEqual({
      error: undefined,
      status: 'fulfilled',
      value: 1,
    });

    // reinitialize because new async fn
    fn = async () => 2;
    rerender(fn);
    expect(result.current).toEqual({
      error: undefined,
      status: 'pending',
      value: undefined,
    });

    // async fn 2 resolves
    await waitForNextUpdate();
    expect(result.current).toEqual({
      error: undefined,
      status: 'fulfilled',
      value: 2,
    });
  });
});
