import type { useCallback } from 'react';
import { useState } from 'react';

/**
 * Stores a value from the initial render and returns that
 * value on every subsequent render, ignoring any future
 * values passed in. This is similar to useMemo with an
 * empty dependency list, but unlike useMemo this hook guarantees
 * the value will remain constant through the life of the component,
 * while useMemo is allowed to forget cached values for performance
 * reasons.
 *
 * @param value initial value or value factory (ignored after first render)
 * @returns The value from the initial render.
 */
export function useConst<V>(value: V | (() => V)): V {
  return useState(value)[0];
}

/**
 * Convenience helper for memoizing lambdas. Equivalent to
 * `useConst(() => callback)` without needing the extra wrapper.
 * This is similar to useCallback with an empty dependency list,
 * but it guarantees the value will remain constant whereas
 * useCallback reserves the right to forget the cached value.
 *
 * @param callback to memoize for the life of the component
 * @returns The memoized callback stored from first render.
 */
export function useConstCallback<C extends Parameters<typeof useCallback>[0]>(
  callback: C,
): C {
  return useState(() => callback)[0];
}
