import { renderHook } from '@testing-library/react-hooks';
import type { ListenerElement, ListenerWindow } from '.';
import { useEvent } from '.';

interface Props {
  handler: (...args: any[]) => void;
  name: string;
  options: any;
  target: ListenerElement | ListenerWindow;
}

describe(useEvent, () => {
  it('should call addEventListener/removeEventListener on mount/unmount', () => {
    const handler = jest.fn();
    const target = {
      addEventListener: jest.fn(),
      removeEventListener: jest.fn(),
    };
    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
    const { unmount } = renderHook(() => useEvent('name', handler, target));

    expect(target.addEventListener).toHaveBeenCalledTimes(1);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
    unmount();
    expect(target.addEventListener).toHaveBeenCalledTimes(1);
    expect(target.removeEventListener).toHaveBeenCalledTimes(1);
  });

  it('should call addEventListener/removeEventListener on deps change', () => {
    const props1: Props = {
      handler: jest.fn(),
      name: 'name',
      options: { a: 'opt1' },
      target: {
        addEventListener: jest.fn(),
        removeEventListener: jest.fn(),
      },
    };
    const props2: Props = {
      handler: jest.fn(),
      name: 'name2',
      options: { a: 'opt2' },
      target: {
        addEventListener: jest.fn(),
        removeEventListener: jest.fn(),
      },
    };
    expect(props1.target.addEventListener).toHaveBeenCalledTimes(0);
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(0);
    const { rerender } = renderHook(
      (p: Props) => useEvent(p.name, p.handler, p.target, p.options),
      {
        initialProps: props1,
      },
    );

    expect(props1.target.addEventListener).toHaveBeenCalledTimes(1);
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(0);

    // Deps are the same
    rerender({
      handler: props1.handler,
      name: props1.name,
      options: props1.options,
      target: props1.target,
    });
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(0);
    expect(props1.target.addEventListener).toHaveBeenCalledTimes(1);

    // name is different
    rerender({
      handler: props1.handler,
      name: props2.name,
      options: props1.options,
      target: props1.target,
    });
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(1);
    expect(props1.target.addEventListener).toHaveBeenCalledTimes(2);

    // handler is different
    rerender({
      handler: props2.handler,
      name: props2.name,
      options: props1.options,
      target: props1.target,
    });
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(2);
    expect(props1.target.addEventListener).toHaveBeenCalledTimes(3);

    // options is different
    rerender({
      handler: props2.handler,
      name: props2.name,
      options: props2.options,
      target: props1.target,
    });
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(3);
    expect(props1.target.addEventListener).toHaveBeenCalledTimes(4);

    // target is different
    rerender({
      handler: props2.handler,
      name: props2.name,
      options: props2.options,
      target: props2.target,
    });
    expect(props1.target.removeEventListener).toHaveBeenCalledTimes(4);
    expect(props1.target.addEventListener).toHaveBeenCalledTimes(4);

    expect(props2.target.addEventListener).toHaveBeenCalledTimes(1);
  });

  it('does nothing if no handler passed', () => {
    const target = {
      addEventListener: jest.fn(),
      removeEventListener: jest.fn(),
    };
    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
    const { unmount } = renderHook(() => useEvent('name', null, target));

    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
    unmount();
    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
  });

  it('does nothing if null target passed', () => {
    const handler = jest.fn();
    const target = {
      addEventListener: jest.fn(),
      removeEventListener: jest.fn(),
    };
    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
    const { unmount } = renderHook(() => useEvent('name', handler, null));

    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
    unmount();
    expect(target.addEventListener).toHaveBeenCalledTimes(0);
    expect(target.removeEventListener).toHaveBeenCalledTimes(0);
  });
});
