import { renderHook } from '@testing-library/react-hooks';
import { useInterval } from '.';

describe(useInterval, () => {
  const callback = jest.fn();

  afterEach(() => {
    callback.mockReset();
  });
  it('calls the callback after the delay, but not before', () => {
    expect(callback).not.toHaveBeenCalled();
    renderHook(() => useInterval(callback, 1000));
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(50);
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(949);
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(1);
    expect(callback).toHaveBeenCalledTimes(1);

    // and again

    jest.advanceTimersByTime(1000);
    expect(callback).toHaveBeenCalledTimes(2);
  });

  it('calls the callback after the specified delay even if the callback changes', () => {
    const callback2 = jest.fn();
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();
    const { rerender } = renderHook((cb) => useInterval(cb, 1000), {
      initialProps: callback,
    });
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();

    jest.advanceTimersByTime(999);
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();

    // switch it on 'em
    rerender(callback2);

    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();

    jest.advanceTimersByTime(1);
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).toHaveBeenCalledTimes(1);

    // switch it on 'em AGAIN
    rerender(callback);

    jest.advanceTimersByTime(1000);
    expect(callback).toHaveBeenCalledTimes(1);
    expect(callback2).toHaveBeenCalledTimes(1);
  });

  it('cleans up after itself', () => {
    expect(callback).not.toHaveBeenCalled();
    const { unmount } = renderHook(() => useInterval(callback, 1000));
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(1000);
    expect(callback).toHaveBeenCalledTimes(1);
    unmount();
    jest.advanceTimersByTime(1000);
    expect(callback).toHaveBeenCalledTimes(1);
  });
});
