import type { MutableRefObject } from 'react';
import { useLayoutEffect, useRef } from 'react';

/**
 * Readonly version of useRef return type for refs that are managed
 * internally and not expected to be set by the caller.
 */
export type ManagedRef<T> = Readonly<MutableRefObject<T>>;

/**
 * Similar to useRef, but updates to the latest passed value in a
 * concurrent safe manner. This means that after the first render
 * the current property will still contain the previous value
 * during the render pass (use the same value passed to the hook
 * directly if needed).
 *
 * @param value to update the ref with
 * @returns A readonly ref object
 */
export function useLatest<V>(value: V): ManagedRef<V> {
  const ref = useRef(value);

  // Direct updates during render are not concurrent safe, use an
  // effect during the commit phase to perform the update instead.
  useBrowserLayoutEffect(() => {
    ref.current = value;
  });

  return ref;
}

// useLayoutEffect throws a warning when rendering to string,
// but we don't really need this to run on first render anyway
const useBrowserLayoutEffect =
  typeof window !== 'undefined' ? useLayoutEffect : () => undefined;
