import type { Dispatch, SetStateAction } from 'react';
import { useState } from 'react';
import { useConst } from '../useConst';
import type { ManagedRef } from '../useLatest';
import { useLatest } from '../useLatest';

/**
 * Function type for the dispatch parameter in CreateMethods and
 * CreateMethodsWithRef (same as the type of the setter returned
 * by useState).
 */
export type MethodDispatch<S> = Dispatch<SetStateAction<S>>;

/**
 * Simple factory function for creating a set of methods that can
 * dispatch state changes.
 */
export type CreateMethods<STATE, METHODS> = (
  dispatch: MethodDispatch<STATE>,
) => METHODS;

/**
 * Advanced factory function for creating a set of methods that can
 * dispatch state changes and also access additional values via ref.
 */
export type CreateMethodsWithRef<STATE, REF, METHODS> = (
  dispatch: MethodDispatch<STATE>,
  ref: ManagedRef<REF>,
) => METHODS;

/**
 * Creates a stable set of methods which can operate on a state
 * object. This can be used similarly to useReducer but with
 * specific methods for each action rather than a shared dispatch
 * that requires action objects to be managed by the caller. In
 * addition, these methods can be complex async operations which
 * dispatch state changes at multiple points before finishing.
 *
 * @param createMethods simple factory function to create a set of
 *  methods bound to a setState dispatch function
 * @param initialState initial state or state factory
 * @returns the current state and a stable set of methods to operate
 *  on that state
 */
export function useMethods<STATE, METHODS>(
  createMethods: CreateMethods<STATE, METHODS>,
  initialState: STATE | (() => STATE),
): [STATE, METHODS];

/**
 * Creates a stable set of methods which can operate on a state
 * object. This can be used similarly to useReducer but with
 * specific methods for each action rather than a shared dispatch
 * that requires action objects to be managed by the caller. In
 * addition, these methods can be complex async operations which
 * dispatch state changes at multiple points before finishing.
 *
 * This version accepts an addition parameter for non-state values
 * that are passed by ref to the factory function for use in the
 * constructed methods. This allows access to data from props or
 * other hooks without needing to explicitly copy those values
 * into the state object.
 *
 * @param createMethods simple factory function to create a set of
 *  methods bound to a setState dispatch function
 * @param initialState initial state or state factory
 * @param values current values of additional non-state data
 * @returns the current state and a stable set of methods to operate
 *  on that state
 */
export function useMethods<STATE, METHODS, REF>(
  createMethods: CreateMethodsWithRef<STATE, REF, METHODS>,
  initialState: STATE | (() => STATE),
  values: REF,
): [STATE, METHODS];

// Common signature for the two overloads above
export function useMethods<STATE, METHODS, REF>(
  createMethods: CreateMethodsWithRef<STATE, REF, METHODS>,
  initialState: STATE | (() => STATE),
  values?: REF,
): [STATE, METHODS] {
  const [state, dispatch] = useState(initialState);
  // Overloads ensure that otherState is only undefined
  // if not used by createMethods, so this cast is safe.
  const ref = useLatest(values as REF);

  return [state, useConst(() => createMethods(dispatch, ref))];
}
