import { act, renderHook } from '@testing-library/react-hooks';
import { useMethods } from '.';

describe(useMethods, () => {
  it('returns constant methods', () => {
    const test = jest.fn();
    const { rerender, result } = renderHook((p) => useMethods(p[0], p[1]), {
      initialProps: [
        () => ({
          test,
        }),
        {},
      ] as const,
    });
    const [initialState, initialMethods] = result.current;
    expect(initialMethods.test).toEqual(test);

    // Re-rendering with new values shouldn't change anything
    act(() =>
      rerender([
        () => ({
          test: jest.fn(),
        }),
        {},
      ] as const),
    );
    expect(result.current[0]).toEqual(initialState);
    expect(result.current[1]).toEqual(initialMethods);
    expect(result.current[1].test).toEqual(test);

    // Calling a method that doesn't dispatch shouldn't change anything
    act(() => initialMethods.test());
    expect(result.current[0]).toEqual(initialState);
    expect(result.current[1]).toEqual(initialMethods);
    expect(result.current[1].test).toEqual(test);
  });

  it('updates state on dispatch', () => {
    const { result } = renderHook(() =>
      useMethods(
        (d) => ({
          inc: jest.fn(() => d((s) => s + 1)),
          set: jest.fn((n) => d(n)),
        }),
        1,
      ),
    );
    const initialMethods = result.current[1];

    // Check initial state
    expect(result.current[0]).toEqual(1);

    // Check basic state update method
    act(() => result.current[1].inc());
    expect(result.current[0]).toEqual(2);
    expect(result.current[1]).toEqual(initialMethods);

    // Methods should be callable more than once
    act(() => {
      result.current[1].inc();
      result.current[1].inc();
    });
    expect(result.current[0]).toEqual(4);
    expect(result.current[1]).toEqual(initialMethods);

    // Check static value setter mthod
    act(() => result.current[1].set(7));
    expect(result.current[0]).toEqual(7);
    expect(result.current[1]).toEqual(initialMethods);

    expect(initialMethods.inc).toHaveBeenCalledTimes(3);
    expect(initialMethods.set).toHaveBeenCalledTimes(1);
  });

  it('updates optional additional ref state', () => {
    const { rerender, result } = renderHook(
      (p) =>
        useMethods(
          (d, r) => ({
            set: jest.fn(() => d(r.current)),
          }),
          0,
          p,
        ),
      {
        initialProps: 1,
      },
    );

    // Initial value should be 1 (ref is 1)
    expect(result.current[0]).toEqual(0);

    // Push ref value into state
    act(() => result.current[1].set());
    expect(result.current[0]).toEqual(1);

    // Re-render with new ref value should not change state
    act(() => rerender(5));
    expect(result.current[0]).toEqual(1);

    // Push ref value into state again
    act(() => result.current[1].set());
    expect(result.current[0]).toEqual(5);

    expect(result.current[1].set).toHaveBeenCalledTimes(2);
  });
});
