import { renderHook } from '@testing-library/react-hooks';
import { usePoll } from '.';

const mockCancel = jest.fn();
jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  poll: () => {
    const fn = () => null;
    fn.cancel = mockCancel;
    return fn;
  },
}));

describe(usePoll, () => {
  beforeEach(() => mockCancel.mockReset());

  it('stops polling when unmounted', () => {
    const { unmount } = renderHook(() =>
      usePoll(jest.fn(), { intervalMs: 100, timeoutMs: Infinity }, []),
    );

    unmount();

    expect(mockCancel).toHaveBeenCalledTimes(1);
  });

  it('stops polling the old fn and generates a new poll fn when fn dependencies changes', () => {
    const firstCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 100 },
      [1],
    ];
    const secondCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 100 },
      [1],
    ];
    const thirdCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 100 },
      [2],
    ];

    // first render
    const { rerender } = renderHook((p) => usePoll(...p), {
      initialProps: firstCall,
    });

    // second render (stable check)
    rerender(secondCall);
    expect(mockCancel).toHaveBeenCalledTimes(0);

    // third render (deps changed)
    rerender(thirdCall);
    expect(mockCancel).toHaveBeenCalledTimes(1);
  });

  it('stops polling the old and generates a new poll fn when the poll options change', () => {
    const firstCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 100 },
      [1],
    ];
    const secondCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 200 },
      [1],
    ];

    // first render
    const { rerender } = renderHook((p) => usePoll(...p), {
      initialProps: firstCall,
    });

    // second render (options changed)
    rerender(secondCall);
    expect(mockCancel).toHaveBeenCalledTimes(1);
  });

  it('continues polling across rerenders', () => {
    const firstCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 100 },
      [1],
    ];
    const secondCall: Parameters<typeof usePoll> = [
      jest.fn(),
      { intervalMs: 100 },
      [1],
    ];

    // first render
    const { rerender } = renderHook((p) => usePoll(...p), {
      initialProps: firstCall,
    });

    // second render
    rerender(secondCall);
    expect(mockCancel).toHaveBeenCalledTimes(0);
  });
});
