import { useState } from 'react';
import { getDefaultStorageProxy } from './storageProxies';

let storage = getDefaultStorageProxy();

export interface ConfigureUseStorageOpts {
  storageProxy: Storage;
}

// istanbul ignore next: trivial
/**
 * An imperative configuration function that allows use of the non-default
 * storage proxy for the `useStorage` hook. It accepts any object that
 * implements the Storage API. It takes a `storageProxy` key.
 *
 * @param param0 configuration object
 */
export function configureUseStorage({
  storageProxy,
}: ConfigureUseStorageOpts): void {
  storage = storageProxy;
}

/**
 * A value or value-providing function. Similar to useState, it offers to accept
 * a function in order to prevent reconstruction of an expensive object inside
 * every invocation of the consumering functional component.
 */
export type DefaultValue<T> = T | (() => T);

/**
 * A hook providing access to a persistent storage backend that caches the value
 * to prevent slowly accessing the storage backend on each render. By default,
 * it uses a `localStorage` proxy that handles JSON serialization to allow
 * storing complex objects; this proxy also gracefully handles situations where
 * `localStorage` throws errors or isn't available and is thus SSR safe. To
 * provide a different storage proxy/backend, use the `configureUseStorage`
 * method.
 *
 * @param key key string identifying your desired storage value
 * @param defaultValue default value to use/persist if one is not already present
 */
export function useStorage<T>(key: string): [T | null, (newValue: T) => void];
export function useStorage<T>(
  key: string,
  defaultValue: DefaultValue<T>,
): [T, (newValue: T) => void];
export function useStorage<T>(
  key: string,
  defaultValue: null,
): [T | null, (newValue: T) => void];
export function useStorage<T>(
  key: string,
  defaultValue: DefaultValue<T> | null = null,
): [T | null, (newValue: T) => void] {
  // the storage proxy is responsible for handling non-string values if
  // users expect that functionality, and so we cast around the interface to
  // the storage access methods
  const [value, setStateValue] = useState(() => {
    const storedValue = storage.getItem(key) as any as T;
    if (storedValue !== null) {
      return storedValue;
    }

    if (defaultValue !== undefined) {
      const actualDefaultValue =
        typeof defaultValue === 'function'
          ? (defaultValue as () => T)()
          : defaultValue;
      // the storage proxy is responsible for handling non-string values if
      // users expect that functionality
      storage.setItem(key, actualDefaultValue as any);
      return actualDefaultValue;
    }

    return null;
  });

  const setValue = (newValue: T) => {
    storage.setItem(key, newValue as any);
    // the storage proxy is responsible for handling non-string values if
    // users expect that functionality
    setStateValue(newValue);
  };

  return [value, setValue];
}
