// istanbul ignore next: trivial
/**
 * A proxy that implements the Storage API but noops all methods to allow for
 * safe usage.
 */
export function getNoopStorageProxy(): Storage {
  return {
    clear(): void {
      return;
    },
    getItem(): null {
      return null;
    },
    key(_index: number): null {
      return null;
    },
    length: 0,
    removeItem(_key: string): void {
      return;
    },
    setItem(_key: string, _value: string): void {
      return;
    },
  };
}

/**
 * Attempts to parse non-null stringified values returned from the localStorage
 * API. If it an encounters an unparseable value (such as a raw string, which
 * might happen if it encounters a value set outside of this framework), it
 * returns the raw value.
 *
 * @param rawValue a value returned from localStorage
 */
function convertRawValue<T>(rawValue: string | null): T | null {
  if (rawValue === null) {
    return null;
  }

  try {
    return JSON.parse(rawValue) as T;
  } catch {
    return rawValue as any as T;
  }
}

/**
 * A proxy around localStorage that handles common error cases and does
 * JSON serialization/deserialization to allow for storing structured data.
 */
export function getSafeLocalStorageProxy(): Storage {
  return {
    clear(): void {
      window.localStorage.clear();
    },
    getItem<T>(key: string): T | null {
      return convertRawValue<T>(window.localStorage.getItem(key));
    },
    key<T>(index: number): T | null {
      return convertRawValue<T>(window.localStorage.key(index));
    },
    get length(): number {
      return window.localStorage.length;
    },
    removeItem(key: string): void {
      window.localStorage.removeItem(key);
    },
    setItem<T>(key: string, value: T): void {
      try {
        window.localStorage.setItem(key, JSON.stringify(value));
      } catch {} // eslint-disable-line no-empty
    },
  };
}

// istanbul ignore next: trivial
export function getDefaultStorageProxy(): Storage {
  try {
    if (typeof window !== 'undefined' && window.localStorage) {
      return getSafeLocalStorageProxy();
    }
  } catch {
    // Fall through to no-op proxy
  }
  return getNoopStorageProxy();
}
