import { datatype, lorem } from 'faker';
import { getDefaultStorageProxy, getSafeLocalStorageProxy } from '.';

const mockLocalStorageGetItem = localStorage.getItem as jest.Mock;
const mockLocalStorageKey = localStorage.key as jest.Mock;
const mockLocalStorageSetItem = localStorage.setItem as jest.Mock;

describe(getDefaultStorageProxy, () => {
  it('returns no-op proxy when localStroage reference throws', () => {
    const mockAccessDenied = jest
      .spyOn(window, 'localStorage', 'get')
      .mockImplementation(() => {
        throw new Error('SecurityError: Access is denied');
      });

    const ret = getDefaultStorageProxy();
    ret.setItem('foo', 'bar');
    expect(mockLocalStorageSetItem).not.toHaveBeenCalled();
    expect(ret.getItem('foo')).toBeNull();

    mockAccessDenied.mockRestore();
  });
});

describe(getSafeLocalStorageProxy, () => {
  const parseableObject = { a: { b: 'c' } };
  const unparseableValue = 'plain string';
  const storageProxy = getSafeLocalStorageProxy();

  beforeEach(() => {
    localStorage.clear();
    (localStorage.clear as jest.Mock).mockReset();
  });

  describe('clear', () => {
    it('passes through to clear()', () => {
      storageProxy.clear();
      expect(localStorage.clear).toHaveBeenCalledTimes(1);
    });
  });

  describe('getItem', () => {
    it('passes key through to getItem()', () => {
      const key = datatype.uuid();
      storageProxy.getItem(key);
      expect(mockLocalStorageGetItem).toHaveBeenLastCalledWith(key);
    });

    it('returns null', () => {
      mockLocalStorageGetItem.mockReturnValueOnce(null);
      expect(storageProxy.getItem(datatype.uuid())).toBeNull();
    });

    it('returns primitives', () => {
      const words = lorem.words();
      mockLocalStorageGetItem.mockReturnValueOnce(words);
      expect(storageProxy.getItem(datatype.uuid())).toEqual(words);
    });

    it('returns complex objects', () => {
      mockLocalStorageGetItem.mockReturnValueOnce(
        JSON.stringify(parseableObject),
      );
      expect(storageProxy.getItem(datatype.uuid())).toEqual(parseableObject);
    });

    it('returns raw unparseable objects', () => {
      mockLocalStorageGetItem.mockReturnValueOnce(unparseableValue);
      expect(storageProxy.getItem(datatype.uuid())).toEqual(unparseableValue);
    });
  });

  describe('key', () => {
    it('passes index through to key()', () => {
      const index = datatype.number();
      storageProxy.key(index);
      expect(mockLocalStorageKey).toHaveBeenLastCalledWith(index);
    });

    it('returns null', () => {
      mockLocalStorageKey.mockReturnValueOnce(null);
      expect(storageProxy.key(datatype.number())).toBeNull();
    });

    it('returns primitives', () => {
      const words = lorem.words();
      mockLocalStorageKey.mockReturnValueOnce(words);
      expect(storageProxy.key(datatype.number())).toEqual(words);
    });

    it('returns complex objects', () => {
      mockLocalStorageKey.mockReturnValueOnce(JSON.stringify(parseableObject));
      expect(storageProxy.key(datatype.number())).toEqual(parseableObject);
    });

    it('returns raw unparseable objects', () => {
      mockLocalStorageKey.mockReturnValueOnce(unparseableValue);
      expect(storageProxy.key(datatype.number())).toEqual(unparseableValue);
    });
  });

  describe('length', () => {
    it('returns length of localStorage', () => {
      expect(storageProxy).toHaveLength(0);
      expect(storageProxy).toHaveLength(localStorage.length);

      localStorage.setItem('a', '1');
      storageProxy.setItem('b', '2');

      expect(storageProxy).toHaveLength(2);
      expect(storageProxy).toHaveLength(localStorage.length);
    });
  });

  describe('removeItem', () => {
    it('passes through to removeItem()', () => {
      const key = datatype.uuid();
      storageProxy.removeItem(key);
      expect(localStorage.removeItem).toHaveBeenLastCalledWith(key);
    });
  });

  describe('setIem', () => {
    it('calls setItem() with stringified value', () => {
      const key = datatype.uuid();
      storageProxy.setItem(key, parseableObject as any);
      expect(localStorage.setItem).toHaveBeenLastCalledWith(
        key,
        JSON.stringify(parseableObject),
      );
    });

    it('does not throw for parse errors', () => {
      expect(() => {
        storageProxy.setItem(datatype.uuid(), unparseableValue as any);
      }).not.toThrow();
    });

    it('does not throw for storage errors', () => {
      mockLocalStorageSetItem.mockImplementationOnce(() => {
        throw new Error();
      });
      expect(() => {
        storageProxy.setItem(datatype.uuid(), parseableObject as any);
      }).not.toThrow();
    });
  });
});
