import { act, renderHook } from '@testing-library/react-hooks';
import { datatype, lorem } from 'faker';
import {
  getDefaultStorageProxy,
  getNoopStorageProxy,
  getSafeLocalStorageProxy,
} from './storageProxies';
import { configureUseStorage, useStorage } from '.';

describe(useStorage, () => {
  afterAll(() => {
    configureUseStorage({ storageProxy: getDefaultStorageProxy() });
  });

  describe('for default storage proxy', () => {
    interface StorageTestCase {
      getValue: () => any;
      title: string;
    }

    // This explicitly doesn't handle Dates, Symbols or Functions since their
    // behavior gets odd very quickly when being stringified. This could be added
    // in the future if the need arises.
    const testCases: StorageTestCase[] = [
      {
        getValue: () => lorem.word(),
        title: 'String',
      },
      {
        getValue: () => ({ [datatype.uuid()]: lorem.word() }),
        title: 'Object',
      },
      {
        getValue: () => ({
          [datatype.uuid()]: [{ [lorem.word()]: lorem.word() }],
        }),
        title: 'Nested Object',
      },
      {
        getValue: () => datatype.number(),
        title: 'Number',
      },
      {
        getValue: () => datatype.boolean(),
        title: 'Boolean',
      },
      {
        getValue: () => null,
        title: 'Null',
      },
    ];

    testCases.forEach(({ getValue, title }) => {
      const expectedValue = getValue();
      let key: string;
      beforeEach(() => {
        key = datatype.uuid();
      });

      afterAll(() => {
        localStorage.clear();
      });

      describe(title, () => {
        describe('when using safe local storage', () => {
          let storage: Storage;
          beforeAll(() => {
            storage = getSafeLocalStorageProxy();
            configureUseStorage({ storageProxy: storage });
          });

          it('gives defaultValue when nothing has been stored (for non-function defaultValue)', () => {
            const {
              result: {
                current: [receivedValue],
              },
            } = renderHook(() => useStorage(key, expectedValue));
            expect(receivedValue).toEqual(expectedValue);
          });

          it('gives defaultValue when nothing has been stored (for function defaultValue)', () => {
            const {
              result: {
                current: [receivedValue],
              },
            } = renderHook(() => useStorage(key, () => expectedValue));
            expect(receivedValue).toEqual(expectedValue);
          });

          // The storage API has no meaningful difference between unset keys and null
          if (expectedValue !== null) {
            it('persists initial defaultValue between sessions and disregards ensuing defaultValues', () => {
              renderHook(() => useStorage(key, expectedValue));
              const {
                result: {
                  current: [receivedSecondValue],
                },
              } = renderHook(() => useStorage(key, lorem.word()));

              expect(receivedSecondValue).toEqual(expectedValue);
            });

            it('gives stored item if one was there beforehand', () => {
              storage.setItem(key, expectedValue);
              const {
                result: {
                  current: [receivedValue],
                },
              } = renderHook(() => useStorage(key, lorem.word()));

              expect(receivedValue).toEqual(expectedValue);
            });
          }

          it('properly overwrites an item', () => {
            const { result } = renderHook(() => useStorage(key, getValue()));
            act(() => result.current[1](expectedValue));
            expect(result.current[0]).toEqual(expectedValue);
          });

          it('persists overwritten item between sessions', () => {
            const {
              result: {
                current: [, setValue],
              },
            } = renderHook(() => useStorage(key, getValue()));

            act(() => setValue(expectedValue));
            const {
              result: {
                current: [receivedValue],
              },
            } = renderHook(() => useStorage(key, getValue()));

            expect(receivedValue).toEqual(expectedValue);
          });
        });

        describe('when using noop storage fallback', () => {
          let storage: Storage;
          beforeAll(() => {
            storage = getNoopStorageProxy();
            configureUseStorage({ storageProxy: storage });
          });

          it('gives defaultValue initially', () => {
            const {
              result: {
                current: [receivedValue],
              },
            } = renderHook(() => useStorage(key, expectedValue));

            expect(receivedValue).toEqual(expectedValue);
          });

          it('allows updating local store', () => {
            const { result } = renderHook(() => useStorage(key, getValue()));
            act(() => result.current[1](expectedValue));
            expect(result.current[0]).toEqual(expectedValue);
          });

          it('local store udpates are not persisted', () => {
            const {
              result: {
                current: [, setValue],
              },
            } = renderHook(() => useStorage(key, getValue()));
            act(() => setValue(getValue()));

            const {
              result: {
                current: [receivedValue],
              },
            } = renderHook(() => useStorage(key, expectedValue));

            expect(receivedValue).toEqual(expectedValue);
          });
        });
      });
    });
  });

  describe('for an arbitrary storage proxy', () => {
    const mockProxy = {
      clear: jest.fn(),
      getItem: jest.fn(),
      key: jest.fn(),
      length: 0,
      removeItem: jest.fn(),
      setItem: jest.fn(),
    };

    beforeAll(() => {
      configureUseStorage({ storageProxy: mockProxy });
    });

    beforeEach(() => {
      mockProxy.getItem.mockReset();
      mockProxy.setItem.mockReset();
    });

    const key = datatype.uuid();
    const value = datatype.uuid();

    it('persists defaultValue when nothing has been stored', () => {
      mockProxy.getItem.mockReturnValue(null);
      renderHook(() => useStorage(key, value));
      expect(mockProxy.getItem).toHaveBeenCalled();
      expect(mockProxy.setItem).toHaveBeenCalledWith(key, value);
    });

    it('persists value when set is called', () => {
      const {
        result: {
          current: [, setValue],
        },
      } = renderHook(() => useStorage(key, datatype.uuid()));
      act(() => setValue(value));
      expect(mockProxy.setItem).toHaveBeenLastCalledWith(key, value);
    });
  });
});
