import { renderHook } from '@testing-library/react-hooks';
import { useTimeout } from '.';

describe(useTimeout, () => {
  const callback = jest.fn();

  afterEach(() => {
    callback.mockReset();
  });

  it('calls the callback on next tick if no delay is passed', () => {
    expect(callback).not.toHaveBeenCalled();
    renderHook(() => useTimeout(callback));
    jest.advanceTimersByTime(0);
    expect(callback).toHaveBeenCalledTimes(1);
  });

  it('calls the callback after the delay, but not before', () => {
    expect(callback).not.toHaveBeenCalled();
    renderHook(() => useTimeout(callback, 1000));
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(50);
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(949);
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(1);
    expect(callback).toHaveBeenCalledTimes(1);
  });

  it('calls the callback after the specified delay even if the callback changes', () => {
    const callback2 = jest.fn();
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();
    const { rerender } = renderHook((cb) => useTimeout(cb, 1000), {
      initialProps: callback,
    });
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();

    jest.advanceTimersByTime(999);
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();

    // switch it on 'em
    rerender(callback2);

    expect(callback).not.toHaveBeenCalled();
    expect(callback2).not.toHaveBeenCalled();

    jest.advanceTimersByTime(1);
    expect(callback).not.toHaveBeenCalled();
    expect(callback2).toHaveBeenCalledTimes(1);
  });

  it('cleans up after itself', () => {
    expect(callback).not.toHaveBeenCalled();
    const { unmount } = renderHook(() => useTimeout(callback, 1000));
    expect(callback).not.toHaveBeenCalled();

    jest.advanceTimersByTime(990);
    expect(callback).not.toHaveBeenCalled();
    unmount();
    jest.advanceTimersByTime(1);
    expect(callback).not.toHaveBeenCalled();
  });
});
