import type { ComponentType } from 'react';
import { Component } from 'react';
import { isBrowser } from 'tachyon-utils-stdlib';
import { getComponentName } from '../../getComponentName';

/**
 * Threshold of how much the element is in view at which to trigger an impresssion
 * 75% is standard across VX impression tracking
 */
const IMPRESSION_THRESHOLD = 0.75;

export type ImpressionCallback = (...args: any[]) => any;

export interface ImpressionListenerProps {
  /** Configure impression listener callbacks */
  impressionListener: {
    /** Register callback to be fired when impression is detected */
    registerImpressionCallback: (
      impressionCallback: ImpressionCallback,
    ) => void;

    /** Unregister callback for when impressions are detected */
    unregisterImpressionCallback: () => void;
  };
}

/**
 * withImpressionListener is a HOC for tracking when the first time a component
 * has an impresssion or becomes visible in the viewport.
 * Impression events are generated using IntersectionObserver.
 * This HOC is disabled for browsers that do not support IntersectionObserver.
 *
 * @param Comp The component to watch impressions for
 */
export function withImpressionListener<P extends ImpressionListenerProps>(
  Comp: ComponentType<P>,
): ComponentType<Omit<P, keyof ImpressionListenerProps>> {
  return class ImpressionListener extends Component<
    Omit<P, keyof ImpressionListenerProps>
  > {
    /** True, if the component has had an impression reported, false otherwise */
    private hadImpression = false;

    /** The callback function for when an impression is being reported */
    private reportImpression: ImpressionCallback | null = null;

    /** IntersectionObserver instance being used to report impressions*/
    private observer: any = null;

    /** The HTML element being observed */
    private observedElement: HTMLElement | null = null;

    public static displayName = `WithImpressionListener(${getComponentName(
      Comp,
    )})`;

    public override componentDidMount(): void {
      const canUseIntersectionObserver =
        isBrowser() &&
        'IntersectionObserver' in window &&
        'IntersectionObserverEntry' in window &&
        'intersectionRatio' in window.IntersectionObserverEntry.prototype;
      if (canUseIntersectionObserver) {
        this.observer = new IntersectionObserver(this.handleObserverUpdate, {
          threshold: IMPRESSION_THRESHOLD,
        });
        this.observer.observe(this.observedElement);
      }
    }

    public override componentWillUnmount(): void {
      if (this.observer) {
        this.observer.disconnect();
      }
    }

    public override render(): JSX.Element {
      const props = {
        ...this.props,
        impressionListener: {
          registerImpressionCallback: this.registerImpressionCallback,
          unregisterImpressionCallback: this.unregisterImpressionCallback,
        },
      } as P;

      return (
        <div ref={this.setRef}>
          <Comp {...props} />
        </div>
      );
    }

    private registerImpressionCallback = (
      reportImpression: ImpressionCallback,
    ): void => {
      this.reportImpression = reportImpression;
    };

    private unregisterImpressionCallback = (): void => {
      this.reportImpression = null;
    };

    private handleObserverUpdate = (
      entries: IntersectionObserverEntry[],
    ): void => {
      if (!this.hadImpression) {
        // Only a single element is observed, so only need to look at first entry
        const entry: IntersectionObserverEntry = entries[0];
        if (entry.isIntersecting) {
          this.hadImpression = true;
          if (this.reportImpression) {
            this.reportImpression();
          }
        }
      }
    };

    private setRef = (el: HTMLElement | null): void => {
      this.observedElement = el;
    };
  };
}
