import { mount } from 'enzyme';
import { Component } from 'react';
import type { ImpressionCallback, ImpressionListenerProps } from '.';
import { withImpressionListener } from '.';

const mockImpressionCallback = jest.fn();
const mockIntersectionObserverDisconnect = jest.fn();
let mockIntersectionObserverCb: ImpressionCallback;

let mockIsBrowser = false;
jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  isBrowser: () => mockIsBrowser,
}));

describe('Impression Listener', () => {
  const TestComponent = withImpressionListener(
    class Comp extends Component<ImpressionListenerProps> {
      public override componentDidMount() {
        this.props.impressionListener.registerImpressionCallback(
          mockImpressionCallback,
        );
      }

      public override render() {
        return <div />;
      }
    },
  );

  beforeAll(() => {
    mockIsBrowser = true;
    (window as any).IntersectionObserver = jest.fn((cb) => {
      mockIntersectionObserverCb = cb;
      const instance = {
        disconnect: mockIntersectionObserverDisconnect,
        observe: jest.fn(),
      };
      return instance;
    });

    (window as any).IntersectionObserverEntry = {
      prototype: {
        intersectionRatio: null,
      },
    };
  });

  describe('functionality', () => {
    it('calls the impression callback only when the component is in view', () => {
      mount(<TestComponent />);

      mockIntersectionObserverCb([{ isIntersecting: false }]);
      expect(mockImpressionCallback).toHaveBeenCalledTimes(0);

      mockIntersectionObserverCb([{ isIntersecting: true }]);
      expect(mockImpressionCallback).toHaveBeenCalledTimes(1);
    });

    it('disconnects the observer when unmounting', () => {
      const wrapper = mount(<TestComponent />);
      wrapper.unmount();
      expect(mockIntersectionObserverDisconnect).toHaveBeenCalledTimes(1);
    });
  });
});
