/**
 * This keeps both the CDN and browsers from caching the resource.
 * https://docs.fastly.com/en/guides/cache-control-tutorial#do-not-cache
 */
export const NoCache = 'private, no-store';

export type BuildNoCacheHeaderOpts = { totalCacheLife: 0 };

export type BuildFullCacheHeaderOpts = {
  staleIfError: number;
  staleWhileRevalidate: number;
  totalCacheLife: number;
};

export type BuildCacheHeaderOpts =
  | BuildFullCacheHeaderOpts
  | BuildNoCacheHeaderOpts;

/**
 * Function that builds a proper cache-header string with `max-age`,
 * `stale-while-revalidate`, `stale-if-error` values. If the `totalCacheLife`
 * option or calculated `max-age` is 0, then will set a CDN-compatible
 * non-caching header.
 */
export function buildCacheHeader(opts: BuildNoCacheHeaderOpts): string;
export function buildCacheHeader(opts: BuildFullCacheHeaderOpts): string;
export function buildCacheHeader(opts: BuildCacheHeaderOpts): string {
  // these 2 cases are identical based on the overload signatures, only
  // including the second case as a typeguard to satisfy TS
  if (opts.totalCacheLife === 0 || !('staleWhileRevalidate' in opts)) {
    return NoCache;
  }

  const { staleIfError, staleWhileRevalidate, totalCacheLife } = opts;

  const maxAge = Math.max(0, totalCacheLife - staleWhileRevalidate);
  if (maxAge === 0) {
    return NoCache;
  }

  return `public, max-age=${maxAge}, stale-while-revalidate=${staleWhileRevalidate}, stale-if-error=${staleIfError}`;
}
