export type PausableIntervalParams = {
  /**
   * Invoked on each interval completed.
   */
  callback: () => void;
  /**
   * Length of the interval delay in milliseconds.
   */
  delayMs: number;
  /**
   * This optional function can be provided to generate a custom initial interval delay.
   * This will be invoked on each "stopAndReset" invocation.
   */
  getFirstDelayMs?: () => number;
  /**
   * The maximum amount of interval drift to account for in the next interval in milliseconds
   * before resetting back to the default interval delay as a safety mechanism. This number should never exceed "delayMS".
   */
  maxDriftBeforeResetMs: number;
};

export type PausableInterval = {
  /**
   * Pauses an interval while saving progress. Calling "start" again will resume
   * the interval with the remaining duration of "delayMs".
   */
  pause: () => void;
  /**
   * Initiates a new interval or resumes from where one was last stopped if there
   * isn't already one in progress.
   * @param callback Invoked when the interval completes.
   */
  start: () => void;
  /**
   * Stops a currently active interval and resets back to an initial state that
   * will use firstDelayMs if provided.
   */
  stopAndReset: () => void;
};

/**
 * A utility for creating long-running "intervals" that can be paused/resumed and have
 * drift correction to account for the single-threaded nature of JavaScript.
 *
 * Interval timing does not begin until "start" is invoked.
 */
export function createPausableInterval({
  callback,
  delayMs,
  getFirstDelayMs,
  maxDriftBeforeResetMs,
}: PausableIntervalParams): PausableInterval {
  let timeoutID: number | null = null;
  let remainingDelayMS = getFirstDelayMs?.() ?? delayMs;
  let expectedIntervalCompletionTime: number | null = null;

  // Ensure this logic remains synchronized with the initialization logic above
  function stopAndReset() {
    if (timeoutID) {
      window.clearTimeout(timeoutID);
      timeoutID = null;
    }

    remainingDelayMS = getFirstDelayMs?.() ?? delayMs;
    expectedIntervalCompletionTime = null;
  }

  function calculateNextIntervalDelay(): number {
    // This should never happen, but if it somehow does, use the default delay
    if (!expectedIntervalCompletionTime) {
      return delayMs;
    }

    const intervalDriftMS = Date.now() - expectedIntervalCompletionTime;

    // If we're off by more than the allowance, or it is negative, this might
    // means something with the user's clock so reset back to using the default
    // delay length.
    if (intervalDriftMS < 0 || intervalDriftMS > maxDriftBeforeResetMs) {
      return delayMs;
    }

    return delayMs - intervalDriftMS;
  }

  function setTimeout(timeoutDurationMs: number): void {
    expectedIntervalCompletionTime = Date.now() + delayMs;
    timeoutID = window.setTimeout(createTimeoutHandler(), timeoutDurationMs);
  }

  function createTimeoutHandler() {
    return () => {
      timeoutID = null;
      callback();

      const delay = calculateNextIntervalDelay();
      setTimeout(delay);
    };
  }

  function start(): void {
    if (timeoutID) {
      return;
    }

    setTimeout(remainingDelayMS);
  }

  function pause(): void {
    if (timeoutID) {
      window.clearTimeout(timeoutID);
      timeoutID = null;
    }

    if (expectedIntervalCompletionTime) {
      remainingDelayMS = expectedIntervalCompletionTime - Date.now();
      expectedIntervalCompletionTime = null;
    } else {
      // This should never happen in practice. If it does, the only sane thing
      // can do is reset the delay.
      remainingDelayMS = delayMs;
    }
  }

  return { pause, start, stopAndReset };
}
