import { datatype } from 'faker';
import type { PausableIntervalParams } from '.';
import { createPausableInterval } from '.';

function setup(overrides?: Partial<PausableIntervalParams>): {
  instance: ReturnType<typeof createPausableInterval>;
  params: PausableIntervalParams;
} {
  const params: PausableIntervalParams = {
    callback: jest.fn(),
    delayMs: datatype.number({ max: 100, min: 60 }),
    maxDriftBeforeResetMs: datatype.number({ max: 20, min: 10 }),
    ...overrides,
  };

  const instance = createPausableInterval(params);

  return { instance, params };
}

describe(createPausableInterval, () => {
  it('invokes and uses getFirstDelayMs for each interval "session"', () => {
    let firstDelayMs = 20;
    const { instance, params } = setup({
      delayMs: 30,
      getFirstDelayMs: () => firstDelayMs,
    });
    instance.start();

    expect(params.callback).not.toHaveBeenCalled();
    jest.advanceTimersByTime(20);
    expect(params.callback).toHaveBeenCalledTimes(1);

    firstDelayMs = params.delayMs + 10;
    instance.stopAndReset();

    instance.start();
    expect(params.callback).toHaveBeenCalledTimes(1);
    jest.advanceTimersByTime(params.delayMs);
    expect(params.callback).toHaveBeenCalledTimes(1);
    jest.advanceTimersByTime(10);
    expect(params.callback).toHaveBeenCalledTimes(2);
  });

  it('respects a first delay length of zero', () => {
    const { instance, params } = setup({
      delayMs: 60,
      getFirstDelayMs: () => 0,
    });
    instance.start();

    expect(params.callback).not.toHaveBeenCalled();
    jest.advanceTimersByTime(1);
    expect(params.callback).toHaveBeenCalledTimes(1);
  });

  describe('when the interval is running', () => {
    it('treats a second start invocation as a no-op', () => {
      const firstDelayMS = datatype.number({ min: 5000 });
      const { instance, params } = setup({
        getFirstDelayMs: () => firstDelayMS,
      });
      instance.start();

      jest.advanceTimersByTime(params.getFirstDelayMs!() - 50);
      instance.start();
      expect(params.callback).not.toHaveBeenCalled();

      jest.advanceTimersByTime(50);
      expect(params.callback).toHaveBeenCalledTimes(1);

      // Make sure that a second timer wasn't started on the second "start"
      // invocation
      jest.advanceTimersByTime(params.delayMs - 1);
      expect(params.callback).toHaveBeenCalledTimes(1);

      jest.advanceTimersByTime(1);
      expect(params.callback).toHaveBeenCalledTimes(2);
    });
  });

  describe('when the interval reaches the end of its provided delay value', () => {
    it('invokes the provided callback at the expect', () => {
      const { instance, params } = setup();

      expect(params.callback).not.toHaveBeenCalled();
      instance.start();

      jest.advanceTimersByTime(params.delayMs - 1);
      expect(params.callback).not.toHaveBeenCalled();
      jest.advanceTimersByTime(1);
      expect(params.callback).toHaveBeenCalledTimes(1);
    });

    it('sets a new interval for the same delay', () => {
      const { instance, params } = setup();

      instance.start();

      jest.advanceTimersByTime(params.delayMs);
      expect(params.callback).toHaveBeenCalledTimes(1);
      jest.advanceTimersByTime(params.delayMs);
      expect(params.callback).toHaveBeenCalledTimes(2);
    });

    it('accounts for drift when setting a subsequent interval delay', () => {
      const { instance, params } = setup();

      instance.start();
      jest.setSystemTime(Date.now() + params.maxDriftBeforeResetMs!);
      jest.advanceTimersByTime(params.delayMs);
      expect(params.callback).toHaveBeenCalledTimes(1);

      jest.advanceTimersByTime(
        params.delayMs - params.maxDriftBeforeResetMs! - 1,
      );
      expect(params.callback).toHaveBeenCalledTimes(1);
      jest.advanceTimersByTime(1);
      expect(params.callback).toHaveBeenCalledTimes(2);
    });

    it('ignores drift when setting a subsequent interval if the interval exceeded the max allowance', () => {
      const { instance, params } = setup();

      instance.start();
      jest.setSystemTime(
        Date.now() + params.delayMs + params.maxDriftBeforeResetMs! + 100,
      );
      jest.advanceTimersByTime(params.delayMs);

      expect(params.callback).toHaveBeenCalledTimes(1);
      jest.advanceTimersByTime(params.delayMs - 1);
      expect(params.callback).toHaveBeenCalledTimes(1);
      jest.advanceTimersByTime(1);
      expect(params.callback).toHaveBeenCalledTimes(2);
    });

    it('sets the exact delayMS if the interval drift was negative', () => {
      const { instance, params } = setup();

      instance.start();
      jest.setSystemTime(Date.now() - 50);
      jest.advanceTimersByTime(params.delayMs);

      expect(params.callback).toHaveBeenCalledTimes(1);
      jest.advanceTimersByTime(params.delayMs - 1);
      expect(params.callback).toHaveBeenCalledTimes(1);
      jest.advanceTimersByTime(1);
      expect(params.callback).toHaveBeenCalledTimes(2);
    });
  });

  describe('when the interval is paused', () => {
    it('invokes the callback after resuming and the required delay occurs', () => {
      const { instance, params } = setup();

      instance.start();
      jest.advanceTimersByTime(params.delayMs - 5);
      instance.pause();
      jest.advanceTimersByTime(10);
      expect(params.callback).not.toHaveBeenCalled();

      instance.start();
      jest.advanceTimersByTime(6);
      expect(params.callback).toHaveBeenCalledTimes(1);
    });
  });
});
