import type { AnyFunction, CancelableFunction } from 'tachyon-type-library';
import type { RequireAtLeastOne } from 'type-fest';

export type ThrottleOptions = RequireAtLeastOne<
  {
    leading: true;
    trailing: true;
  },
  'leading' | 'trailing'
>;

export type DebounceOptions = ThrottleOptions & {
  maxWait?: number;
};

interface DebouncedFunc<Fn extends AnyFunction> extends CancelableFunction {
  (...args: Parameters<Fn>): ReturnType<Fn>;
}

export type Debounce = <Fn extends AnyFunction>(
  func: Fn,
  wait: number,
  options: DebounceOptions,
) => DebouncedFunc<Fn>;

function noop(name: string): () => number {
  return () => {
    // eslint-disable-next-line no-console
    console.warn(
      `You invoked ${name} in a server side context, which is a noop`,
    );
    return -1;
  };
}

/**
 * Creates a debounced function that delays invoking `func` until after `wait`
 * milliseconds have elapsed since the last time the debounced function was
 * invoked, or until the next browser frame is drawn. The debounced function
 * comes with a `cancel` method to cancel delayed `func` invocations. Provide
 * `options` to indicate whether `func` should be invoked on the leading
 * and/or trailing edge of the`wait` timeout. The `func` is invoked with the
 * last arguments provided to the debounced function. Subsequent calls to the
 * debounced function return the result of the last `func` invocation.
 *
 * **Note:** If `leading` and `trailing` options are `true`, `func` is
 * invoked on the trailing edge of the timeout only if the debounced function
 * is invoked more than once during the `wait` timeout.
 *
 * If `wait` is `0` and `leading` is `false`, `func` invocation is deferred
 * until the next tick, similar to `setTimeout` with a timeout of `0`.
 *
 * If `wait` is omitted in an environment with `requestAnimationFrame`, `func`
 * invocation will be deferred until the next frame is drawn (typically about
 * 16ms).
 *
 * See [David Corbacho's article](https://css-tricks.com/debouncing-throttling-explained-examples/)
 * for details over the differences between `debounce` and `throttle`.
 *
 * @param {Function} func The function to debounce.
 * @param {number} [wait=0]
 *  The number of milliseconds to delay
 * @param {Object} [options={}] The options object.
 * @param {boolean} [options.leading=false]
 *  Specify invoking on the leading edge of the timeout.
 * @param {number} [options.maxWait]
 *  The maximum time `func` is allowed to be delayed before it's invoked.
 * @param {boolean} [options.trailing=true]
 *  Specify invoking on the trailing edge of the timeout.
 * @returns {Function} Returns the new debounced function.
 */

export function debounce<Fn extends AnyFunction>(
  func: Fn,
  wait: number,
  options: DebounceOptions,
): DebouncedFunc<Fn> {
  let timerId: number | undefined;
  let lastCallTime: number;
  let lastInvokeTime: number;
  let lastArgs: any[] | undefined;
  let lastReturn: any;
  const timeout =
    typeof window === 'undefined' ? noop('setTimeout') : window.setTimeout;
  const clear =
    typeof window === 'undefined' ? noop('clearTimeout') : window.clearTimeout;

  function remainingWait(time: number) {
    const timeSinceLastCall = time - lastCallTime;
    const timeSinceLastInvoke = time - lastInvokeTime;
    const timeWaiting = wait - timeSinceLastCall;

    return options.maxWait
      ? Math.min(timeWaiting, options.maxWait - timeSinceLastInvoke)
      : timeWaiting;
  }

  function invokeFunc(time: number) {
    if (lastArgs === undefined) {
      return;
    }

    lastInvokeTime = time;
    lastReturn = func(...lastArgs);
    lastArgs = undefined;
  }

  function timerExpired() {
    const time = Date.now();
    invokeFunc(time);
  }

  function debounced(...args: Parameters<Fn>) {
    const time = Date.now();
    lastArgs = args;

    if ('leading' in options && (!lastCallTime || remainingWait(time) <= 0)) {
      invokeFunc(time);
    }

    if ('trailing' in options) {
      clear(timerId);
      const timeSinceLastInvoke = time - lastInvokeTime;
      timerId = timeout(
        timerExpired,
        options.maxWait ? options.maxWait - timeSinceLastInvoke : wait,
      );
    }

    lastCallTime = time;

    return lastReturn;
  }

  function cancel() {
    if (timerId !== undefined) {
      clear(timerId);
    }
  }

  debounced.cancel = cancel;

  return debounced;
}

/**
 * Creates a throttled function that only invokes `func` at most once
 * per every `wait` milliseconds. See debounce for a detailed explanation.
 */
export const throttle: Debounce = (func, wait, options: ThrottleOptions) =>
  debounce(func, wait, { ...options, maxWait: wait });
