import { debounce, throttle } from '.';

const WAIT = 1000;

describe(debounce, () => {
  it('leading edge', () => {
    const mockedFn = jest.fn();
    const debouncedFn = debounce(mockedFn, WAIT, {
      leading: true,
    });

    expect(mockedFn).toHaveBeenCalledTimes(0);
    debouncedFn();
    expect(mockedFn).toHaveBeenCalledTimes(1);

    // invoke debounced fn several times with less than WAIT
    // time between invocations
    for (let i = 0; i < 10; i++) {
      jest.advanceTimersByTime(WAIT / 2);
      debouncedFn();
    }
    expect(mockedFn).toHaveBeenCalledTimes(1);

    // does not call on the trailing edge
    jest.advanceTimersByTime(WAIT);
    expect(mockedFn).toHaveBeenCalledTimes(1);

    // should call on leading edge again now that WAIT time has elapsed
    debouncedFn();
    expect(mockedFn).toHaveBeenCalledTimes(2);
  });

  it('trailing edge', () => {
    const mockedFn = jest.fn();
    const debouncedFn = debounce(mockedFn, WAIT, {
      trailing: true,
    });

    // invoke debounced fn several times with less than WAIT
    // time between invocations
    for (let i = 0; i < 10; i++) {
      jest.advanceTimersByTime(WAIT / 2);
      debouncedFn();
    }
    expect(mockedFn).toHaveBeenCalledTimes(0);
    // allow WAIT time to elapse
    jest.advanceTimersByTime(WAIT);
    expect(mockedFn).toHaveBeenCalledTimes(1);
  });

  it('cancels', () => {
    const mockedFn = jest.fn();
    const debouncedFn = debounce(mockedFn, WAIT, {
      trailing: true,
    });

    debouncedFn();
    debouncedFn.cancel();
    jest.advanceTimersByTime(WAIT);

    expect(mockedFn).toHaveBeenCalledTimes(0);
  });
});

describe(throttle, () => {
  it('leading edge maxWait', () => {
    const mockedFn = jest.fn();
    const throttledFn = throttle(mockedFn, WAIT, {
      leading: true,
      maxWait: WAIT,
    });

    // invoke debounced fn several times with less than WAIT
    // time between invocations
    for (let i = 0; i < 10; i++) {
      jest.advanceTimersByTime(WAIT / 2);
      throttledFn();
    }
    // should have been called once per WAIT time interval
    expect(mockedFn).toHaveBeenCalledTimes(5);
  });

  it('trailing edge maxWait', () => {
    const mockedFn = jest.fn();
    const throttledFn = throttle(mockedFn, WAIT, {
      maxWait: WAIT,
      trailing: true,
    });

    // invoke debounced fn several times with less than WAIT
    // time between invocations
    for (let i = 0; i < 10; i++) {
      jest.advanceTimersByTime(WAIT / 2);
      throttledFn();
    }

    // called once per WAIT time interval
    expect(mockedFn).toHaveBeenCalledTimes(5);
  });

  it('cancels', () => {
    const mockedFn = jest.fn();
    const throttledFn = throttle(mockedFn, WAIT, {
      maxWait: WAIT,
      trailing: true,
    });

    throttledFn();
    throttledFn.cancel();
    jest.advanceTimersByTime(WAIT);

    expect(mockedFn).toHaveBeenCalledTimes(0);
  });
});
