import type { Merge } from 'type-fest';
import { isPlainObject } from '../isPlainObject';

type ObjectWithPotentialNesting = Record<string, unknown>;

/**
 * Recursively assigns default properties to a new object, without mutating
 * the target object in the process.
 *
 * - Primitive override types such as string, number, boolean, or Symbol always
 *   replace the default's equivalent key
 * - Default keys containing Arrays or ES6 classes are replaced rather than merged
 * - Default keys containing POJOs are merged if the override's key is also a POJO

 * NOTE: The root is a special case, in that if you pass in an empty object
 * as the new object, no overrides will take place and you will get the original
 * object back. Within the tree however, an empty object will completely clobber
 * from the corresponding subtree. To the greatest extent possible, the types
 * try to model this behavior but some situations will require using generics
 * (recommended) or explicit casts.
 *
 * @param defaults the default object
 * @param overrides the object to merge on top of defaults
 * @returns object with newly merged features
 */
export function defaultsDeep<
  Defaults extends ObjectWithPotentialNesting,
  Overrides extends ObjectWithPotentialNesting,
  Output extends Merge<Defaults, Overrides> = Merge<Defaults, Overrides>,
>(defaults: Defaults, overrides: Overrides): Output {
  return _defaultsDeep(defaults, overrides, true);
}

/**
 * Private implementation to allow tracking whether invocation is root call
 * without exposing that argument in the public API
 */
function _defaultsDeep<
  Defaults extends ObjectWithPotentialNesting,
  Overrides extends ObjectWithPotentialNesting,
  Output extends Merge<Defaults, Overrides> = Merge<Defaults, Overrides>,
>(defaults: Defaults, overrides: Overrides, root: boolean): Output {
  if (!root && Object.keys(overrides).length === 0) {
    return {} as Output;
  }

  const output = { ...defaults } as ObjectWithPotentialNesting;
  Object.entries(overrides).forEach(([key, newValue]) => {
    if (isPlainObject(newValue) && isPlainObject(defaults[key])) {
      output[key] = _defaultsDeep(
        defaults[key] as ObjectWithPotentialNesting,
        newValue,
        false,
      );
    } else {
      output[key] = newValue;
    }
  });

  return output as Output;
}
