import { defaultsDeep } from '.';

describe(defaultsDeep, () => {
  type DefaultsDeepTestCase = {
    defaults: {};
    expected: {};
    overrides: {};
  };

  it.each`
    defaults                                                        | overrides                             | expected
    ${{}}                                                           | ${{ bar: true }}                      | ${{ bar: true }}
    ${{ bar: { baz: true } }}                                       | ${{}}                                 | ${{ bar: { baz: true } }}
    ${{ bar: { baz: true } }}                                       | ${{ bar: {} }}                        | ${{ bar: {} }}
    ${{ bar: true }}                                                | ${{ bar: false }}                     | ${{ bar: false }}
    ${{ bar: [1] }}                                                 | ${{ bar: [] }}                        | ${{ bar: [] }}
    ${{ bar: [1] }}                                                 | ${{ bar: [2] }}                       | ${{ bar: [2] }}
    ${{ bar: 31 }}                                                  | ${{ bar: 42 }}                        | ${{ bar: 42 }}
    ${{ bar: 'hi' }}                                                | ${{ bat: 'hello', boing: 'foo' }}     | ${{ bar: 'hi', bat: 'hello', boing: 'foo' }}
    ${{ bar: 'value' }}                                             | ${{ bar: null }}                      | ${{ bar: null }}
    ${{ bar: null }}                                                | ${{ baz: null }}                      | ${{ bar: null, baz: null }}
    ${{ bar: null }}                                                | ${{ bar: 'value' }}                   | ${{ bar: 'value' }}
    ${{ 12: 'hello' }}                                              | ${{ 12: 'hello again' }}              | ${{ 12: 'hello again' }}
    ${{ bar: { prop1: null } }}                                     | ${{ bar: { prop2: null } }}           | ${{ bar: { prop1: null, prop2: null } }}
    ${{ dateField: new Date(1990) }}                                | ${{ dateField: new Date(2004) }}      | ${{ dateField: new Date(2004) }}
    ${{ bar: { baz: { boing: 'hey', foo: 'hi' }, boo: 'bye' } }}    | ${{ bar: { baz: undefined } }}        | ${{ bar: { baz: undefined, boo: 'bye' } }}
    ${{ bar: { baz: 'boop', foo: 'hi' } }}                          | ${{ bar: { baz: undefined } }}        | ${{ bar: { baz: undefined, foo: 'hi' } }}
    ${{ bar: { baz: [1, 2, 3] } }}                                  | ${{ bar: { baz: [] } }}               | ${{ bar: { baz: [] } }}
    ${{ bar: { baz: { boing: 'hey', foo: 'hello' }, boo: 'bye' } }} | ${{ bar: undefined }}                 | ${{ bar: undefined }}
    ${{ bar: 'hi', baz: { boing: 'hey', foo: 'boop' } }}            | ${{ bar: undefined, baz: undefined }} | ${{ bar: undefined, baz: undefined }}
  `(
    'returns $expected when $overrides is added to $target',
    ({ defaults, expected, overrides }: DefaultsDeepTestCase) => {
      expect(defaultsDeep(defaults, overrides)).toEqual(expected);
    },
  );

  // Symbols are unique and don't work with the Jest's each syntax
  it('implementation works with Symbols', () => {
    const person = { bar: Symbol('Derek') };
    const person2 = { bar: Symbol('Simmy') };

    expect(defaultsDeep(person, person2)).toEqual(person2);
  });

  it('does not mutate input objects', () => {
    const item = {
      foo: {
        bar: {
          property: 'immutable',
        },
      },
    };
    const item2 = {
      foo: {
        bar: {
          property2: 'immutable2',
        },
      },
    };

    expect(defaultsDeep(item, item2)).toEqual({
      foo: {
        bar: {
          property: 'immutable',
          property2: 'immutable2',
        },
      },
    });
    expect(item).toEqual({
      foo: {
        bar: {
          property: 'immutable',
        },
      },
    });
    expect(item2).toEqual({
      foo: {
        bar: {
          property2: 'immutable2',
        },
      },
    });
  });

  // eslint-disable-next-line jest/no-disabled-tests
  it.skip('type tests, checked by typechecker', () => {
    // function to test type alignment
    function testArgIsOfType<T>(x: T): boolean {
      return !x;
    }

    // SIMPLE OBJECTS
    type A = { a: number };
    type B = { b: number };
    type C = { c: number };
    type AB = A & B;
    type AC = A & C;

    const a: A = { a: 1 };
    const b: B = { b: 1 };

    testArgIsOfType<AB>(defaultsDeep(a, b));
    // @ts-expect-error: recognizes missing top-level key
    testArgIsOfType<AC>(defaultsDeep(a, b));

    // SINGLY-NESTED OBJECTS
    type D_A = { d: A };
    type D_B = { d: B };
    type D_AB = { d: AB };
    type D_AC = { d: AC };

    const d_a: D_A = { d: a };
    const d_b: D_B = { d: b };

    testArgIsOfType<D_AB>(defaultsDeep(d_a, d_b));
    // @ts-expect-error: recognizes missing first-level nested key
    testArgIsOfType<D_AC>(defaultsDeep(d_a, d_b));

    // MULTI_NESTED OBJECTS
    type E_D_A = { e: D_A };
    type E_D_B = { e: D_B };
    type E_D_AB = { e: D_AB };
    type E_D_AC = { e: D_AC };

    const e_d_a: E_D_A = { e: d_a };
    const e_d_b: E_D_B = { e: d_b };

    testArgIsOfType<E_D_AB>(defaultsDeep(e_d_a, e_d_b));
    // @ts-expect-error: recognizes missing second-level nested key
    testArgIsOfType<E_D_AC>(defaultsDeep(e_d_a, e_d_b));

    // SIMPLE OVERRIDES
    type Astring = { a: string };
    type AstringB = Astring & B;

    const aStringB: AstringB = { a: 'a', b: 1 };

    testArgIsOfType<AB>(defaultsDeep(aStringB, a));
    // @ts-expect-error: recognizes number replaced string
    testArgIsOfType<AstringB>(defaultsDeep(aStringB, a));

    // NESTED OVERRIDES
    type D_AstringB = { d: AstringB };

    const d_aStringB: D_AstringB = { d: aStringB };

    testArgIsOfType<D_AB>(defaultsDeep(d_aStringB, d_a));
    // @ts-expect-error: recognizes number replaced string in nested object
    testArgIsOfType<D_AstringB>(defaultsDeep(d_aStringB, d_a));

    // REPLACING OBJECTS
    type D = { d: number };

    const d: D = { d: 1 };

    testArgIsOfType<D>(defaultsDeep(d_a, d));
    // @ts-expect-error: recognizes object replaced number
    testArgIsOfType<D_A>(defaultsDeep(d_a, d));

    testArgIsOfType<D_A>(defaultsDeep(d, d_a));
    // @ts-expect-error: recognizes number replaced object
    testArgIsOfType<D>(defaultsDeep(d, d_a));
  });
});
