/**
 * All valid formats `fetchData` supports.
 */
export enum Formats {
  JSON = 'json',
  TEXT = 'text',
}

/**
 * The default timeout for `fetchData` when run on the server.
 * No timeout is used on the client.
 */
export const DEFAULT_SERVER_TIMEOUT = 1500;

/**
 * Fetch data from a given `url` with `headers` and then parse the response
 * assuming it is of type `format`. On the server wait a max of `timeout` and
 * throw if the response does not succeed. If it succeeds return the parsed
 * data.
 *
 * @param url The URL to fetch data from.
 * @param headers Headers to send to the server with the request (default: none).
 * @param format The expected content type of the data (default: JSON).
 * @param timeout Maximum wait time for response (server only) (default: `DEFAULT_SERVER_TIMEOUT`).
 */
export async function fetchData<ResponseT>(
  url: string,
  headers: { [key: string]: string } = {},
  format: Formats = Formats.JSON,
  timeout: number = DEFAULT_SERVER_TIMEOUT,
): Promise<ResponseT> {
  const fetchOptions = {
    headers: {
      ...headers,
    },
    timeout,
  };
  const response = await fetch(url, fetchOptions);
  if (!response.ok) {
    throw response;
  }

  switch (format) {
    case Formats.JSON:
      return response.json();
    case Formats.TEXT:
    default:
      return response.text() as any;
  }
}
