export type HeaderOrParamValue =
  | Array<string | {}>
  | string
  | {}
  | null
  | undefined;

/**
 * Function that accommodates flattening headers or query params, since they can
 * be either string or string[]. Function defaults to picking the first element
 * of an array, but can also be set to pick the last element with the optional
 * useLast parameter. For any query params that are more complex (objects, etc),
 * this will return undefined; this behavior is only relevant in server
 * contexts.
 *
 * @param {string|string[]|object|object[]|undefined} value a header or query param value
 * @param {boolean} [useLast=false] optional parameter to use last element if array
 * @return {string|undefined} the flattened value
 */
export function flattenHeaderOrParam(
  value: HeaderOrParamValue,
  useLast = false,
): string | undefined {
  if (value === undefined || value === null) {
    return;
  }

  if (typeof value === 'string') {
    return value;
  }

  if (!Array.isArray(value)) {
    return;
  }

  const valueFromArray = useLast ? value[value.length - 1] : value[0];
  return typeof valueFromArray === 'string' ? valueFromArray : undefined;
}
