import { titleCase } from '.';

describe('titleCase', () => {
  describe('noop cases', () => {
    it('does not die on an empty string', () => {
      expect(titleCase('')).toEqual('');
    });

    it('does not care about numbers', () => {
      expect(titleCase('1')).toEqual('1');
    });

    it('does not care about symbols', () => {
      expect(titleCase('#')).toEqual('#');
    });

    it('does not care about emoji', () => {
      expect(titleCase('🙄')).toEqual('🙄');
    });
  });

  describe('single word', () => {
    it('capitalizes a single letter', () => {
      expect(titleCase('a')).toEqual('A');
    });

    it('does nothing on already capitalized letter', () => {
      expect(titleCase('A')).toEqual('A');
    });

    it('works on characters with diacritics', () => {
      expect(titleCase('ç')).toEqual('Ç');
    });

    it('capitalizes the first letter of a word', () => {
      expect(titleCase('bigAndy')).toEqual('BigAndy');
    });

    it('does nothing when word first letter is already capitalized', () => {
      expect(titleCase('BiggerAndy')).toEqual('BiggerAndy');
    });
  });

  describe('multi word', () => {
    it('capitalizes a series of single letters', () => {
      expect(titleCase('a b c d e')).toEqual('A B C D E');
    });

    it('handles intermingled capital and non-capital letters', () => {
      expect(titleCase('a B c D e')).toEqual('A B C D E');
    });

    it('capitalizes a series of lowercase-starting words', () => {
      expect(titleCase('always be careful doing everything')).toEqual(
        'Always Be Careful Doing Everything',
      );
    });

    it('handles intermingled uppercase- and lowercase-starting words', () => {
      expect(titleCase('always be careful Doing Everything')).toEqual(
        'Always Be Careful Doing Everything',
      );
    });

    it('handles diacritics in multiple words', () => {
      expect(titleCase('ça ñino')).toEqual('Ça Ñino');
    });
  });

  describe('combo', () => {
    it('handles a mix of stuff', () => {
      expect(
        titleCase(
          "😩 i cannot believe 😩 that i've written 😩 15 tests for this 😩 stupid function 😩",
        ),
      ).toEqual(
        "😩 I Cannot Believe 😩 That I've Written 😩 15 Tests For This 😩 Stupid Function 😩",
      );
    });
  });
});
