/**
 * Values correspond to "key" KeyboardEvent values. For applications that predate
 * KeyValue support, use the legacyKeyboardEventHandler.
 */
export enum KeyValue {
  Down = 'ArrowDown',
  Enter = 'Enter',
  Escape = 'Escape',
  Left = 'ArrowLeft',
  NextTV = 'NextTV',
  PauseTV = 'PauseTV',
  PlayTV = 'PlayTV',
  PrevTV = 'PrevTV',
  Right = 'ArrowRight',
  Up = 'ArrowUp',
}

/**
 * Maps "key" to legacy "keyCode" for KeyboardEvent values. Do not use directly,
 * instead use KeyValue plus "legacyKeyboardEventHandler".
 */
export enum KeyCode {
  Enter = 13,
  Escape = 27,
  ArrowUp = 38,
  ArrowDown = 40,
  ArrowLeft = 37,
  ArrowRight = 39,
  PauseTV = 19,
  PlayTV = 415,
  PrevTV = 412,
  NextTV = 417,
}

/**
 * list of keycodes that need to have KeyValues attached for WebOS. These
 * need to map to matching KeyCode values and KeyValue values.
 */
export enum TVCodes {
  PauseTV = 19,
  PlayTV = 415,
  PrevTV = 412,
  NextTV = 417,
}

/**
 * Inspects a KeyBoard event to determine if it "key" is present. If not, it
 * will insert it by inspecting "keyCode" and comparing against known matching
 * "key" values.
 *
 * This is necessary work around for:
 * - 2017 LG (webOS 3.5 / Chromium 38)
 */
export function legacyKeyboardEventHandler(
  callback: (event: KeyboardEvent) => void,
): (event: KeyboardEvent) => void {
  return (event: KeyboardEvent): void => {
    if (
      !('key' in event) ||
      typeof event.key === 'undefined' ||
      event.keyCode in TVCodes
    ) {
      // We set key to be the empty string so that it doesn't match
      // but also won't explode if a client tries destructuring event
      (event.key as any) = KeyCode[event.keyCode] ?? '';
    }
    callback(event);
  };
}
