/**
 * A higher order function that returns a new function that only executes the
 * inputed function the first `count` times.
 *
 * @param count The number of times to allow `func` to be called.
 * @param func The function to limit calls on.
 */
function callMaxTimes<T>(
  count: number,
  func: (...args: any[]) => T,
): (...args: any[]) => T {
  let result: T;

  return (...args: any[]): T => {
    if (count > 0) {
      count--;
      result = func(...args);
    }

    return result;
  };
}

/**
 * A higher order function that wraps `func` to guarantee it is only ever
 * called once.
 *
 * @param func The function to only call once.
 * @returns A version of `func` that only performs the logic once.
 */
export function once<T>(func: (...args: any[]) => T): (...args: any[]) => T {
  return callMaxTimes(1, func);
}
