import type { AnyFunction, CancelableFunction } from 'tachyon-type-library';
import type { PromiseValue } from 'type-fest';

function sleep(ms: number): Promise<void> {
  return new Promise((resolve) => {
    setTimeout(resolve, ms);
  });
}

export interface PollingFn<Fn extends AnyFunction> extends CancelableFunction {
  (...args: Parameters<Fn>): Promise<PromiseValue<ReturnType<Fn>>>;
}

export const CANCELLED_PROMISE_ERROR_MESSAGE = 'canceled promise';
export const CANCELLED_PROMISE_ERROR = new Error(
  CANCELLED_PROMISE_ERROR_MESSAGE,
);

export type PollOpts = {
  intervalMs: number;
  timeoutMs?: number;
};

/**
 * Creates a polling function that repeatedly invokes `fn` every `intervalMs`
 * until fn sucessfully returns, is cancelled, or `timeoutMs` elapses.
 *
 * @param {Function} fn The function to poll.
 * @param {number} [intervalMs]
 *  The number of milliseconds to wait between invokations of fn.
 * @param {number} [timeoutMs=Infinity] The total number of milliseconds to
 * poll.
 */
export function poll<Fn extends AnyFunction>(
  fn: Fn,
  { intervalMs, timeoutMs = Infinity }: PollOpts,
): PollingFn<Fn> {
  const stop = Date.now() + timeoutMs;
  let lastError: unknown;
  let cancelled = false;

  async function pollingFn(...args: Parameters<Fn>) {
    while (Date.now() <= stop) {
      if (cancelled) {
        throw CANCELLED_PROMISE_ERROR;
      }

      try {
        return await fn(...args);
      } catch (e) {
        lastError = e;
      }
      await sleep(intervalMs);
    }

    throw lastError;
  }

  pollingFn.cancel = () => {
    cancelled = true;
  };

  return pollingFn;
}
