import { CANCELLED_PROMISE_ERROR, poll } from '.';

async function advancePromisesAndTimers(ms: number): Promise<void> {
  // https://stackoverflow.com/questions/51126786/jest-fake-timers-with-promises/51132058#51132058
  await Promise.resolve();
  jest.advanceTimersByTime(ms);
}

describe(poll, () => {
  it("returns fn's result when successful", async () => {
    const fn = jest.fn(async () => 3);
    const polledFn = poll(fn, { intervalMs: 1000 });

    await expect(polledFn()).resolves.toBe(3);
  });

  it("returns fn's last error when timing out", async () => {
    const error = new Error('1');
    const fn = jest.fn(async () => {
      throw error;
    });

    const timeoutMs = 1;

    const polledFn = poll(fn, { intervalMs: 0, timeoutMs });
    const retVal = polledFn();

    await advancePromisesAndTimers(0);
    await advancePromisesAndTimers(timeoutMs + 1);

    expect(fn).toHaveBeenCalledTimes(2);
    await expect(retVal).rejects.toEqual(error);
  });

  it('repeatedly invokes the function every intervalMs', async () => {
    const fn = jest.fn();
    fn.mockImplementationOnce(async () => {
      throw new Error();
    })
      .mockImplementationOnce(async () => {
        throw new Error();
      })
      .mockImplementationOnce(async () => 3);

    const intervalMs = 1000;
    const polledFn = poll(fn, { intervalMs });
    const retVal = polledFn();
    await advancePromisesAndTimers(intervalMs);
    await advancePromisesAndTimers(intervalMs);

    expect(fn).toHaveBeenCalledTimes(3);
    await expect(retVal).resolves.toEqual(3);
  });

  it('stops polling when canceled', async () => {
    const fn = jest.fn();
    fn.mockImplementationOnce(async () => {
      throw new Error();
    })
      .mockImplementationOnce(async () => {
        throw new Error();
      })
      .mockImplementationOnce(async () => 3);

    const intervalMs = 1000;
    const polledFn = poll(fn, { intervalMs });
    const retVal = polledFn();
    await advancePromisesAndTimers(intervalMs);
    polledFn.cancel();
    await advancePromisesAndTimers(intervalMs);

    expect(fn).toHaveBeenCalledTimes(2);
    await expect(retVal).rejects.toEqual(CANCELLED_PROMISE_ERROR);
  });
});
