import {
  ensureBoolean,
  ensureNumber,
  ensureNumberArray,
  ensureString,
  ensureStringArray,
} from '.';

type TestCase<T = any> = {
  ensureFunc: (input: unknown, fallback: T) => T;
  fallback: T;
  invalidCases: any[];
  validCases: T[];
};

describe.each`
  ensureFunc           | validCases                   | invalidCases       | fallback
  ${ensureBoolean}     | ${[false, true]}             | ${['false', 1]}    | ${true}
  ${ensureNumber}      | ${[0, 1, -1, 1.5]}           | ${['1', true]}     | ${1}
  ${ensureString}      | ${['', '1', 'true']}         | ${[true, 0, 1]}    | ${'fallback'}
  ${ensureNumberArray} | ${[[0], [1, 2], []]}         | ${[[true], ['0']]} | ${[99, 100]}
  ${ensureStringArray} | ${[[''], ['1', 'true'], []]} | ${[[true], [0]]}   | ${['a', 'b']}
`(
  '$ensureFunc',
  ({ ensureFunc, fallback, invalidCases, validCases }: TestCase) => {
    validCases.forEach((valid, index) => {
      it(`yields the passed-in value when of the right type (${index})`, () => {
        expect(ensureFunc(valid, fallback)).toEqual(valid);
      });
    });

    invalidCases.forEach((invalid, index) => {
      it(`yields the fallback when passed-in value is wrong type (${index})`, () => {
        expect(ensureFunc(invalid, fallback)).toEqual(fallback);
      });
    });

    it('yields fallback when passed undefined', () => {
      expect(ensureFunc(undefined, fallback)).toEqual(fallback);
    });

    it('yields fallback when passed null', () => {
      expect(ensureFunc(null, fallback)).toEqual(fallback);
    });
  },
);
