/**
 * Utility functions for working with enums.
 */
export class Enum {
  /**
   * Return all keys from an enumeration.
   *
   * @param enumType The enumeration type.
   */
  public static keys<T extends Enum>(enumType: T): Array<keyof T> {
    const keys = Object.keys(enumType) as Array<keyof T>;
    return keys.filter((key) => {
      // any-cast to avoid `keyof T` gymnastics since we're only doing
      // comparisons and not logic
      const value: any = enumType[key];
      return (
        // if value is not in keys, we know it's not a number enum
        // for number enums, the reverse lookups are dropped (eg "1": "FOO" === "FOO")
        // while the original values are allowed due to strict comparison (eg "FOO": 1 !== "1")
        !keys.includes(value) ||
        // allow the pattern where the enum value is the same as key (eg FOO: "FOO")
        key === value
      );
    });
  }

  /**
   * Return all values from an enumeration.
   *
   * @param enumType The enumeration type.
   */
  public static values<T extends Enum>(enumType: T): Array<T[keyof T]> {
    return Enum.keys(enumType).map(
      (key) => enumType[key as keyof typeof enumType],
    );
  }

  /**
   * Return all entries from an enumeration.
   *
   * @param enumType The enumeration type.
   */
  public static entries<T extends Enum>(
    enumType: T,
  ): Array<[keyof T, T[keyof T]]> {
    return Enum.keys(enumType).map((key) => [
      key,
      enumType[key as keyof typeof enumType],
    ]);
  }

  /**
   * Tests if a value is in an enumeration.
   *
   * @param enumType The enumeration type.
   * @param value The test value.
   */
  public static includes<T extends Enum>(
    enumType: T,
    value: T[keyof T],
  ): boolean {
    return Enum.values(enumType).includes(value);
  }

  /**
   * Returns an enumeration type from an external value (treating strings as
   * case-insensitive). Returns undefined if value is not found. Accepts null
   * and undefined for the value param for usage convenience.
   *
   * @param enumType The enumeration type.
   * @param value The value to normalize to an enumeration type.
   */
  public static convertValueFromExternal<T extends Enum>(
    enumType: T,
    value: number | string | null | undefined,
  ): T[keyof T] | undefined {
    return Enum.values(enumType).reduce<T[keyof T] | undefined>((acc, v) => {
      if (
        // @ts-expect-error comparing external to generic
        v === value ||
        (typeof v === 'string' &&
          typeof value === 'string' &&
          v.toLowerCase() === value.toLowerCase())
      ) {
        return v;
      }

      return acc;
    }, undefined);
  }
}
