import { Enum } from '.';

describe(Enum, () => {
  enum DefaultValues {
    I0,
    I1,
  }

  enum ExplicitNumbers {
    I2 = 2,
    I3 = 3,
  }

  enum ExplicitStrings {
    I4 = '4',
    I5 = 'five',
  }

  enum ExplicitNumbersAndStrings {
    I6 = 6,
    I7 = '7',
  }

  enum MatchingKeyAndValueStrings {
    I8 = 'I8',
    I9 = 'I9',
  }

  describe(Enum.keys, () => {
    interface EnumKeysTestCase {
      keys: string[];
      testEnum: Enum;
    }

    it.each`
      testEnum                      | keys
      ${DefaultValues}              | ${['I0', 'I1']}
      ${ExplicitNumbers}            | ${['I2', 'I3']}
      ${ExplicitStrings}            | ${['I4', 'I5']}
      ${ExplicitNumbersAndStrings}  | ${['I6', 'I7']}
      ${MatchingKeyAndValueStrings} | ${['I8', 'I9']}
    `(
      'correctly generates keys $keys',
      ({ keys, testEnum }: EnumKeysTestCase) => {
        expect(Enum.keys(testEnum)).toEqual(keys);
      },
    );
  });

  describe(Enum.values, () => {
    interface EnumValuesTestCase {
      testEnum: Enum;
      values: Array<number | string>;
    }

    it.each`
      testEnum                      | values
      ${DefaultValues}              | ${[0, 1]}
      ${ExplicitNumbers}            | ${[2, 3]}
      ${ExplicitStrings}            | ${['4', 'five']}
      ${ExplicitNumbersAndStrings}  | ${[6, '7']}
      ${MatchingKeyAndValueStrings} | ${['I8', 'I9']}
    `(
      'correctly generates values $values',
      ({ testEnum, values }: EnumValuesTestCase) => {
        expect(Enum.values(testEnum)).toEqual(values);
      },
    );
  });

  describe(Enum.entries, () => {
    interface EnumValuesTestCase {
      testEnum: Enum;
      values: Array<number | string>;
    }

    it.each`
      testEnum                      | values
      ${DefaultValues}              | ${[['I0', 0], ['I1', 1]]}
      ${ExplicitNumbers}            | ${[['I2', 2], ['I3', 3]]}
      ${ExplicitStrings}            | ${[['I4', '4'], ['I5', 'five']]}
      ${ExplicitNumbersAndStrings}  | ${[['I6', 6], ['I7', '7']]}
      ${MatchingKeyAndValueStrings} | ${[['I8', 'I8'], ['I9', 'I9']]}
    `(
      'correctly generates values $values',
      ({ testEnum, values }: EnumValuesTestCase) => {
        expect(Enum.entries(testEnum)).toEqual(values);
      },
    );
  });

  describe(Enum.includes, () => {
    interface EnumIncludesTestCase {
      result: boolean;
      testEnum: Enum;
      value: number | string;
    }

    it.each`
      testEnum                      | value     | result
      ${DefaultValues}              | ${0}      | ${true}
      ${DefaultValues}              | ${'0'}    | ${false}
      ${ExplicitNumbers}            | ${2}      | ${true}
      ${ExplicitNumbers}            | ${'2'}    | ${false}
      ${ExplicitStrings}            | ${'4'}    | ${true}
      ${ExplicitStrings}            | ${'five'} | ${true}
      ${ExplicitStrings}            | ${4}      | ${false}
      ${ExplicitStrings}            | ${5}      | ${false}
      ${ExplicitNumbersAndStrings}  | ${6}      | ${true}
      ${ExplicitNumbersAndStrings}  | ${'7'}    | ${true}
      ${ExplicitNumbersAndStrings}  | ${'6'}    | ${false}
      ${ExplicitNumbersAndStrings}  | ${7}      | ${false}
      ${MatchingKeyAndValueStrings} | ${'I8'}   | ${true}
      ${MatchingKeyAndValueStrings} | ${'I9'}   | ${true}
    `(
      'correctly tests $value and gets $result',
      ({ result, testEnum, value }: EnumIncludesTestCase) => {
        expect(Enum.includes(testEnum, value as never)).toEqual(result);
      },
    );
  });

  describe(Enum.convertValueFromExternal, () => {
    it('correctly finds a value when it is present', () => {
      expect(Enum.convertValueFromExternal(ExplicitStrings, 'five')).toEqual(
        ExplicitStrings.I5,
      );
    });

    it('matches case-insensitively', () => {
      expect(Enum.convertValueFromExternal(ExplicitStrings, 'FIVE')).toEqual(
        ExplicitStrings.I5,
      );
    });

    it('returns undefined when value is not present', () => {
      expect(
        Enum.convertValueFromExternal(ExplicitStrings, '3'),
      ).toBeUndefined();
    });

    it('returns undefined when value is undefined', () => {
      expect(
        Enum.convertValueFromExternal(ExplicitStrings, undefined),
      ).toBeUndefined();
    });

    it('returns undefined when value is null', () => {
      expect(
        Enum.convertValueFromExternal(ExplicitStrings, null),
      ).toBeUndefined();
    });

    it('correctly finds a value when it is present in an enum containing non-string values', () => {
      expect(
        Enum.convertValueFromExternal(ExplicitNumbersAndStrings, 6),
      ).toEqual(ExplicitNumbersAndStrings.I6);
    });
  });
});
