import type { CookieSerializeOptions } from 'cookie';
import { parse, serialize } from 'cookie';
import type { JsonObject } from 'type-fest';

const SECONDS_IN_DAY = 60 * 60 * 24;
// 390 days (~13 months) per updated legal policies
export const MAX_COOKIE_AGE = 390 * SECONDS_IN_DAY;

/**
 * Cookies that are explictly not namespaces and used across client / application boundaries.
 */
export const GLOBAL_COOKIE_NAMES = {
  /**
   * The standard key for the auth token cookie across twitch platforms.
   */
  AUTH_TOKEN_COOKIE_NAME: 'auth-token',

  /**
   * The standard key for the device cookie across twitch platforms.
   *
   * More information about device IDs can be found at the Tracking Users doc page:
   * https://git.xarth.tv/pages/emerging-platforms/tachyon/d/apps/tomorrow/processes/tracking-users/
   */
  DEVICE_ID_COOKIE_NAME: 'unique_id',
};

export const DEFAULT_COOKIE_DOMAIN = 'twitch.tv';
let cookieDomain = DEFAULT_COOKIE_DOMAIN;

/**
 * Chrome 51 - 66 will silently drop cookies that set SameSite to unknown values.
 * https://www.chromium.org/updates/same-site/incompatible-clients
 *
 * This impacts us because we set SameSite=None.
 */
export const SAMESITE_COMPAT = '_samesite_compat';

/**
 * Sets the default domain used when setting cookies (individual usages of
 * setCookieValue can still override this default setting). Original value
 * is `twitch.tv`.
 *
 * @param newDomain The domain that cookies will be set to.
 */
export function setCookieDomain(newDomain: string): void {
  cookieDomain = newDomain;
}

/**
 * Resets the default domain used when setting cookies to `twitch.tv`
 * (individual of setCookieValue can still override this default setting).
 */
export function resetCookieDomain(): void {
  cookieDomain = DEFAULT_COOKIE_DOMAIN;
}

/**
 * We enforce that any records stored via a cookie include a version key so that schemas can
 * be safely evolved over time.
 */
export type VersionedPayload = JsonObject & {
  /**
   * The version for the payload. This enables safe changing of the payload shape.
   */
  version: number | string;
};

export function getComplexValue<Payload extends VersionedPayload>(
  value: string | undefined,
): Payload | undefined {
  if (value) {
    try {
      return JSON.parse(value);
    } catch {
      return undefined;
    }
  }
}

export function setComplexValue<Payload extends VersionedPayload>(
  value: Payload,
): string {
  return JSON.stringify(value);
}

export interface GetCookieValueOpts {
  /**
   * The name of the desired cookie value
   */
  name: string;
}

/**
 * Retrieves a value from the cookie string by name. Takes a name, and optional migrationNames.
 * When storing a value other than a string (such as JSON / Objects) use `getCookieComplexValue`.
 *
 * NOTE: `getAndExtendCookieValue` should generally be preferred over using this utility directly.
 *
 * @param param0 The cookie for which the value is being retrieved.
 */
export function getCookieValue({
  name,
}: GetCookieValueOpts): string | undefined {
  return (
    parse(window.document.cookie)[name] ??
    parse(window.document.cookie)[name + SAMESITE_COMPAT]
  );
}

/**
 * Identical to `getCookieValue` but is used when an object was stored instead of a string. The object is deserialized.
 *
 * NOTE: `getAndExtendCookieComplexValue` should generally be preferred over using this utility directly.
 *
 * @param param0 The cookie for which the value is being retrieved.
 */
export function getCookieComplexValue<Payload extends VersionedPayload>(
  opts: GetCookieValueOpts,
): Payload | undefined {
  return getComplexValue(getCookieValue(opts));
}

export type GetAndExtendCookieValueOpts = GetCookieValueOpts & {
  extension?: number;
  /**
   * Previous names for this cookie. This is used to migrate cookie names, eg to namespace them.
   */
  migrationNames?: string[];
};

/**
 * Retrieves a value from the cookie string by name and also extends the cookie expiration. Takes a name, and optional migrationNames.
 * When storing a value other than a string (such as JSON / Objects) use `getAndExtendCookieComplexValue`.
 */
export function getAndExtendCookieValue({
  extension = MAX_COOKIE_AGE,
  migrationNames = [],
  name,
}: GetAndExtendCookieValueOpts): string | undefined {
  for (const cookieName of [name, ...migrationNames]) {
    const value = getCookieValue({ name: cookieName });
    if (value) {
      setCookieValue({ name, opts: { maxAge: extension }, value });
      // remove migrationName cookie because we've replaced it with the new name
      if (cookieName !== name) {
        clearCookieValue({ name: cookieName });
      }
      return value;
    }
  }
}

/**
 * Identical to `getAndExtendCookieValue` but is used when an object was stored instead of a string. The object is deserialized.
 *
 */
export function getAndExtendCookieComplexValue<
  Payload extends VersionedPayload,
>(opts: GetAndExtendCookieValueOpts): Payload | undefined {
  return getComplexValue(getAndExtendCookieValue(opts));
}

/**
 * Generates cookie options while setting defaults for domain (currently
 * configured cookie domain), path ('/'), same site ('none'), and secure (true).
 * All options can be overriden, but should only be done sparringly.
 */
export function generateCookieOpts(
  opts: CookieSerializeOptions = {},
): CookieSerializeOptions {
  return {
    domain: cookieDomain,
    maxAge: MAX_COOKIE_AGE,
    path: '/',
    // default to None so that cookies work in iframe/embedded situations
    // (especially important for inter-op with Twilight; None is their default)
    sameSite: 'none',
    secure: true,
    ...opts,
  };
}

export interface SetCookieValueOpts {
  name: string;
  opts?: Omit<CookieSerializeOptions, 'encode'>;
  value: string;
}

/**
 * Sets cookie by name with a specific value. In addition to name and value,
 * takes an opts object for configuring the various cookie options; path will
 * default to '/', domain will default to the current default cookie domain, and
 * maxAge is mandatory.
 *
 * When storing a value other than a string (such as JSON / Objects) use `setCookieComplexValue`.
 *
 * @param param0 Config object with name, value, and cookie options.
 */
export function setCookieValue({
  name,
  opts,
  value,
}: SetCookieValueOpts): void {
  const options = generateCookieOpts(opts);
  window.document.cookie = serialize(name, value, options);

  if (options.sameSite === 'none') {
    const { sameSite, ...optionsCompat } = options;
    window.document.cookie = serialize(
      `${name}${SAMESITE_COMPAT}`,
      value,
      optionsCompat,
    );
  }
}

export type SetCookieValueComplexOpts = Omit<SetCookieValueOpts, 'value'> & {
  value: VersionedPayload;
};

/**
 * Identical to `setCookieValue` but is used when storing an object instead of a string. The object is serialized and can be retrieved via `getAndExtendCookieComplexValue` or `getCookieComplexValue`.
 */
export function setCookieComplexValue(opts: SetCookieValueComplexOpts): void {
  return setCookieValue({ ...opts, value: setComplexValue(opts.value) });
}

export type ClearCookieOpts = Omit<
  CookieSerializeOptions,
  'encode' | 'expires' | 'maxAge'
>;

export interface ClearCookieValueOpts {
  name: string;
  opts?: ClearCookieOpts;
}

/**
 * Clears a cookie by name. In addition to name, takes an opts object for
 * configuring the various cookie options; path will default to '/' and domain
 * will default to the current default cookie domain.
 *
 * This will always attempt to delete both the base version and _samesite_compat
 * versions of the cookie to ensure that neither one remains even if sameSite
 * options have somehow changed.
 *
 * @param param0 Config object with name and cookie options.
 */
export function clearCookieValue({
  name,
  opts = {},
}: ClearCookieValueOpts): void {
  [name, `${name}${SAMESITE_COMPAT}`].forEach((cookieName) => {
    setCookieValue({
      name: cookieName,
      opts: {
        ...opts,
        expires: new Date(0),
        maxAge: 0,
      },
      value: '',
    });
  });
}
