import type { CookieSerializeOptions } from 'cookie';
import { parse, serialize } from 'cookie';
import { datatype, internet, random } from 'faker';
import type { ClearCookieOpts } from '.';
import {
  DEFAULT_COOKIE_DOMAIN,
  MAX_COOKIE_AGE,
  SAMESITE_COMPAT,
  clearCookieValue,
  generateCookieOpts,
  getAndExtendCookieComplexValue,
  getAndExtendCookieValue,
  getComplexValue,
  getCookieComplexValue,
  getCookieValue,
  resetCookieDomain,
  setComplexValue,
  setCookieComplexValue,
  setCookieDomain,
  setCookieValue,
} from '.';

jest.mock('cookie', () => ({
  parse: jest.fn(),
  serialize: jest.fn(),
}));
const mockParse = parse as jest.Mock;
const mockSerialize = serialize as jest.Mock;

describe('cookies', () => {
  afterAll(() => {
    resetCookieDomain();
    // @ts-expect-error: tests
    delete window.document.cookie;
  });

  beforeEach(() => {
    resetCookieDomain();
    // @ts-expect-error: tests
    delete window.document.cookie;
    mockParse.mockImplementation(() => ({}));
    mockSerialize.mockImplementation(() => '');
  });

  const name = random.alphaNumeric();
  const value = random.alphaNumeric();

  describe(getComplexValue, () => {
    it('returns undefined when JSON parsing fails', () => {
      expect(getComplexValue('foo')).toBeUndefined();
    });
  });

  describe(getCookieValue, () => {
    it('works with just a name when it is present', () => {
      const cookies = { [name]: value };
      mockParse.mockImplementation(() => cookies);

      expect(getCookieValue({ name })).toEqual(value);
    });

    it('returns undefined when no matching values', () => {
      expect(getCookieValue({ name })).toBeUndefined();
    });

    it('fallsback to sameSite compatability cookie when name is not present', () => {
      const cookies = { [name + SAMESITE_COMPAT]: value };
      mockParse.mockImplementation(() => cookies);

      expect(getCookieValue({ name })).toEqual(value);
    });
  });

  describe(getCookieComplexValue, () => {
    it('retrieves and deserializes values', () => {
      const complex = { foo: 'bar', version: 1 };
      const cookies = { [name]: setComplexValue(complex) };
      mockParse.mockImplementation(() => cookies);

      expect(getCookieComplexValue({ name })).toEqual(complex);
    });
  });

  describe(getAndExtendCookieValue, () => {
    it('extends the cookie value when present', () => {
      const cookies = { [name]: value };
      mockParse.mockImplementation(() => cookies);

      expect(getAndExtendCookieValue({ name })).toEqual(value);
      expect(mockSerialize).toHaveBeenCalledWith(
        name,
        value,
        expect.objectContaining({ maxAge: MAX_COOKIE_AGE }),
      );
    });

    it('does not extend the cookie value when missing', () => {
      const cookies = {};
      mockParse.mockImplementation(() => cookies);

      expect(getAndExtendCookieValue({ name })).toBeUndefined();
      expect(mockSerialize).not.toHaveBeenCalled();
    });

    it('replaces the migrationName cookie with the new name', () => {
      const oldName = 'foo';
      const cookies = { [oldName]: value };
      mockParse.mockImplementation(() => cookies);

      expect(
        getAndExtendCookieValue({ migrationNames: [oldName], name }),
      ).toEqual(value);
      expect(mockSerialize).toHaveBeenCalledWith(
        name,
        value,
        expect.objectContaining({ maxAge: MAX_COOKIE_AGE }),
      );
      expect(mockSerialize).toHaveBeenCalledWith(
        oldName,
        '',
        expect.objectContaining({
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });

    it('fallsback to migrationNames', () => {
      const namespaced = `tachyon-${name}`;
      const cookies = { [name]: value };
      mockParse.mockImplementation(() => cookies);

      expect(
        getAndExtendCookieValue({ migrationNames: [name], name: namespaced }),
      ).toEqual(value);
    });

    it('uses name over migrationNames when name is present', () => {
      const namespaced = `tachyon-${name}`;
      const cookies = { [name]: value, [namespaced]: 'foo' };
      mockParse.mockImplementation(() => cookies);

      expect(
        getAndExtendCookieValue({ migrationNames: [name], name: namespaced }),
      ).toEqual('foo');
    });
  });

  describe(getAndExtendCookieComplexValue, () => {
    it('retrieves and deserializes values', () => {
      const complex = { foo: 'bar', version: 1 };
      const cookies = { [name]: setComplexValue(complex) };
      mockParse.mockImplementation(() => cookies);

      expect(getAndExtendCookieComplexValue({ name })).toEqual(complex);
    });
  });

  describe(generateCookieOpts, () => {
    it('sets defaults and passes through non-defaulted values', () => {
      const opts: CookieSerializeOptions = {
        httpOnly: true,
      };

      expect(generateCookieOpts(opts)).toEqual({
        ...opts,
        domain: DEFAULT_COOKIE_DOMAIN,
        maxAge: MAX_COOKIE_AGE,
        path: '/',
        sameSite: 'none',
        secure: true,
      });
    });

    it('overrides all defaults with explicit values', () => {
      const opts: CookieSerializeOptions = {
        domain: internet.url(),
        maxAge: 5,
        path: random.alphaNumeric(),
        sameSite: 'strict',
        secure: false,
      };

      expect(generateCookieOpts(opts)).toEqual(opts);
    });

    it('uses an explicitly set cookie domain', () => {
      const domain = internet.url();
      setCookieDomain(domain);

      expect(generateCookieOpts().domain).toEqual(domain);
    });
  });

  describe(setCookieValue, () => {
    it('sets a cookie', () => {
      const output = random.alphaNumeric();
      mockSerialize.mockImplementation(() => output);
      const opts = { maxAge: datatype.number() };
      setCookieValue({ name, opts, value });

      expect(mockSerialize).toHaveBeenCalledWith(
        name,
        value,
        generateCookieOpts(opts),
      );

      expect(window.document.cookie).toEqual(output);
    });

    it('sets a sameSite compatability cookie when sameSite is set to none', () => {
      const output = random.alphaNumeric();
      mockSerialize.mockImplementation(() => output);
      const opts = { maxAge: datatype.number() };
      setCookieValue({ name, opts, value });
      const options = generateCookieOpts(opts);
      delete options.sameSite;

      expect(mockSerialize).toHaveBeenCalledWith(
        name + SAMESITE_COMPAT,
        value,
        options,
      );

      expect(window.document.cookie).toEqual(output);
    });
  });

  describe(setCookieComplexValue, () => {
    it('sets the serialized value', () => {
      const complex = { foo: 'bar', version: 1 };

      setCookieComplexValue({ name, value: complex });

      expect(mockSerialize).toHaveBeenCalledWith(
        name,
        setComplexValue(complex),
        generateCookieOpts(),
      );
    });
  });

  describe(clearCookieValue, () => {
    it('removes a cookie with default opts', () => {
      clearCookieValue({ name });

      expect(mockSerialize).toHaveBeenCalledWith(
        name,
        '',
        expect.objectContaining({
          domain: DEFAULT_COOKIE_DOMAIN,
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });

    it('removes a cookie with explicit opts with explicitly set cookie domain', () => {
      const domain = internet.url();
      setCookieDomain(domain);

      const opts: ClearCookieOpts = {
        httpOnly: true,
        path: random.alphaNumeric(),
        sameSite: 'strict',
      };
      clearCookieValue({ name, opts });

      expect(mockSerialize).toHaveBeenCalledWith(
        name,
        '',
        expect.objectContaining({
          ...opts,
          domain,
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });

    it('removes the sameSite compatability cookie', () => {
      clearCookieValue({ name });

      expect(mockSerialize).toHaveBeenCalledWith(
        name + SAMESITE_COMPAT,
        '',
        expect.objectContaining({
          domain: DEFAULT_COOKIE_DOMAIN,
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });

    it('removes the sameSite compatability cookie even when sameSite is not none', () => {
      clearCookieValue({ name, opts: { sameSite: 'lax' } });

      expect(mockSerialize).toHaveBeenCalledWith(
        name + SAMESITE_COMPAT,
        '',
        expect.objectContaining({
          domain: DEFAULT_COOKIE_DOMAIN,
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });
  });
});
